jQuery(function ($) {
  const $province  = () => $('#billing_state');       // Province
  const $city      = () => $('#billing_city');        // Municipality / City
  const $barangay  = () => $('#billing_address_1');   // Barangay

  let locations = null;
  let jsonLoaded = false;

  // ---- helpers ----
  const PROVINCE_ALIASES = {
    // aliases → canonical JSON key
    'ncr': 'Metro Manila',
    'metro manila': 'Metro Manila',
    'national capital region': 'Metro Manila',
    'metro manila (ncr)': 'Metro Manila',
    'manila': 'Metro Manila', // optional convenience
  };

  function canon(str) {
    return (str || '')
      .toString()
      .trim()
      .replace(/\s+/g, ' ')
      .toLowerCase();
  }

  function mapProvinceAlias(input) {
    const c = canon(input);
    if (PROVINCE_ALIASES[c]) return PROVINCE_ALIASES[c];
    return input && input.trim() ? input.trim() : '';
  }

  function loadJSON(cb) {
    if (jsonLoaded && locations) return cb && cb();
    $.getJSON(locationData.jsonUrl)
      .done(function (data) { locations = data; jsonLoaded = true; cb && cb(); })
      .fail(function () { console.error('Failed to load locations JSON:', locationData.jsonUrl); });
  }

  function resetSelect($el, placeholder) {
    $el.empty().append(new Option(placeholder, ''));
    if ($el.hasClass('select2-hidden-accessible')) $el.trigger('change.select2');
  }

  // Prefer the visible text (usually matches your JSON keys), fall back to value.
  function getSelectedProvinceKey() {
    const $p = $province();
    const text = $p.find('option:selected').text();
    const val  = $p.val();
    // normalize with alias map
    let key = mapProvinceAlias(text) || mapProvinceAlias(val);
    // final safety: if not found in JSON, try case-insensitive match over JSON keys
    if (key && locations && !locations[key]) {
      const want = canon(key);
      const found = Object.keys(locations).find(k => canon(k) === want);
      if (found) key = found;
    }
    return key;
  }

  function populateCities(provinceKey) {
    const $c = $city(), $b = $barangay();

    resetSelect($c, 'Select a city/municipality');
    resetSelect($b, 'Select a barangay');

    if (!provinceKey || !locations || !locations[provinceKey]) {
      if (provinceKey && locations && !locations[provinceKey]) {
        console.warn('Province not found in JSON:', provinceKey, 'Available keys:', Object.keys(locations));
      }
      return;
    }

    const cities = locations[provinceKey];
    Object.keys(cities).forEach(name => $c.append(new Option(name, name)));
    $c.trigger('change');
  }

  function populateBarangays(provinceKey, cityName) {
    const $b = $barangay();
    resetSelect($b, 'Select a barangay');

    if (!provinceKey || !cityName || !locations || !locations[provinceKey] || !locations[provinceKey][cityName]) return;

    const list = locations[provinceKey][cityName];
    list.forEach(brgy => $b.append(new Option(brgy, brgy)));
    $b.trigger('change');
  }

  function bindHandlers() {
    // delegated so it survives Woo checkout fragment swaps
    $(document.body)
      .off('change.pcb', '#billing_state')
      .on('change.pcb', '#billing_state', function () {
        populateCities(getSelectedProvinceKey());
      });

    $(document.body)
      .off('change.pcb', '#billing_city')
      .on('change.pcb', '#billing_city', function () {
        populateBarangays(getSelectedProvinceKey(), $(this).val());
      });

    // If Select2 is used
    $(document.body)
      .off('select2:select.pcb', '#billing_state')
      .on('select2:select.pcb', '#billing_state', function () {
        populateCities(getSelectedProvinceKey());
      });

    $(document.body)
      .off('select2:select.pcb', '#billing_city')
      .on('select2:select.pcb', '#billing_city', function () {
        populateBarangays(getSelectedProvinceKey(), $(this).val());
      });
  }

  function initialPopulateIfNeeded() {
    const $p = $province(), $c = $city(), $b = $barangay();
    if (!$p.length || !$c.length || !$b.length) return;

    const prov = getSelectedProvinceKey();

    if (prov) {
      populateCities(prov);

      // preserve restored selections (after validation failure, etc.)
      const cityVal = $c.val();
      if (cityVal) {
        $c.val(cityVal).trigger('change'); // triggers barangay populate
        const brgyVal = $b.val();
        if (brgyVal) $b.val(brgyVal).trigger('change');
      }
    } else {
      resetSelect($c, 'Select a city/municipality');
      resetSelect($b, 'Select a barangay');
    }
  }

  function init() {
    bindHandlers();
    loadJSON(initialPopulateIfNeeded);
  }

  // run now and after Woo fragments refresh
  init();
  $(document.body).on('updated_checkout', init);
});
