<?php

namespace App\Addons\Multivendor\Http\Controllers\Admin;

use Hash;
use Artisan;
use App\Models\City;
use App\Models\Shop;
use App\Models\User;
use App\Models\Order;
use App\Models\State;
use App\Models\Region;
use App\Models\Product;
use App\Models\Barangay;
use App\Models\OrderDetail;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\SellerPackagePayment;
use App\Notifications\EmailVerificationNotification;
use App\Notifications\SellerAccountApprovedNotification;
use Illuminate\Support\Facades\Schema;

class SellerController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_seller_products'])->only('seller_products');
    $this->middleware(['permission:show_seller_orders'])->only('seller_orders');
    $this->middleware(['permission:show_sellers'])->only('all_sellers');
  }

  public function all_sellers(Request $request)
  {
    $sort_search = null;
    $approved    = null;

    $sellers = User::query()
      ->where('user_type', 'seller')
      ->with(['seller_package', 'sellerPackagePayment']);

    if ($request->filled('approved_status')) {
      $approved = (int) $request->approved_status;
      $sellers->where('approval', $approved);
    }

    if ($request->filled('search')) {
      $sort_search = trim($request->search);
      $sellers->where(function ($q) use ($sort_search) {
        $q->where('name', 'like', "%{$sort_search}%")
          ->orWhere('phone', 'like', "%{$sort_search}%")
          ->orWhere('email', 'like', "%{$sort_search}%");
      });
    }

    $sellers = $sellers
      ->latest('id')
      ->paginate(15)
      ->appends($request->query());

    foreach ($sellers as $seller) {
      $seller->assigned_name = $this->resolveAssignedLocation($seller);
    }

    return view('addon:multivendor::admin.sellers.index', [
      'sellers'     => $sellers,
      'sort_search' => $sort_search,
      'approved'    => $approved,
    ]);
  }

  public function seller_create()
  {
    return view('addon:multivendor::admin.sellers.create');
  }

  public function seller_store(Request $request)
  {
    if (User::where('email', $request->email)->first() != null) {
      flash(translate('Email already exists!'))->error();
      return back();
    }
    if ($request->password == $request->confirm_password) {
      $user = new User;
      $user->name = $request->name;
      $user->email = $request->email;
      $user->phone = $request->phone;
      $user->user_type = "seller";
      $user->password = Hash::make($request->password);

      if ($user->save()) {
        if (get_setting('email_verification') != 1) {
          $user->email_verified_at = date('Y-m-d H:m:s');
        } else {
          $user->notify(new EmailVerificationNotification());
        }
        $user->save();

        $shop = new Shop;
        $shop->user_id = $user->id;
        $shop->name = $request->shop_name;
        $shop->slug = preg_replace('/\s+/', '-', $request->shop_name) . '-' . $shop->id;
        $shop->save();

        flash(translate('Seller has been added successfully'))->success();
        return redirect()->route('admin.all_sellers');
      }
      flash(translate('Something went wrong'))->error();
      return back();
    } else {
      flash("Password and confirm password didn't match")->warning();
      return back();
    }
  }

  public function seller_edit($id)
  {
    $seller = User::findOrFail(decrypt($id));
    return view('addon:multivendor::admin.sellers.edit', compact('seller'));
  }

  public function seller_update(Request $request)
  {
    $user = User::findOrFail($request->seller_id);
    $user->name = $request->name;
    $user->email = $request->email;
    if (User::where('id', '!=', $user->id)->where('email', $request->email)->first() == null) {
      if (strlen($request->password) > 0) {
        $user->password = Hash::make($request->password);
      }
      if ($user->save()) {
        flash(translate('Seller has been updated successfully'))->success();
        return redirect()->route('admin.all_sellers');
      }

      flash(translate('Something went wrong'))->error();
      return back();
    } else {
      flash(translate('Email Already Exists!'))->error();
      return back();
    }
  }

  public function seller_destroy($id)
  {
    $user = User::findOrFail($id);

    $approvalStatus = SellerPackagePayment::where('user_id', $user->id)->value('approval');

    if ($approvalStatus == 0) {
      $shopId = Shop::where('user_id', $user->id)->value('id');

      Barangay::where('seller_assign', $shopId)->update(['seller_assign' => null]);
      City::where('seller_assign', $shopId)->update(['seller_assign' => null]);
      State::where('seller_assign', $shopId)->update(['seller_assign' => null]);
      Region::where('seller_assign', $shopId)->update(['seller_assign' => null]);

      Shop::where('user_id', $user->id)->delete();
      if (User::destroy($id)) {
        flash(translate('Seller has been deleted successfully'))->success();
        return redirect()->route('admin.all_sellers');
      } else {
        flash(translate('Something went wrong'))->error();
        return back();
      }
    } else {
      Shop::where('user_id', $user->id)->update(['approval' => 0]);

      flash(translate('User approval has been updated'))->success();
      return redirect()->route('admin.all_sellers');
    }
  }

  public function update_seller_approval(Request $request)
  {
    $request->validate([
      'id'     => 'required|integer',
      'status' => 'required|in:0,1',
    ]);

    $userId = (int) $request->id;
    $turnOn = (int) $request->status === 1;

    $user = User::findOrFail($userId);

    // Use latest payment's package to decide if payment is required (match your working app)
    if ($turnOn) {
      $requiresPayment = (int) optional(
        SellerPackagePayment::with('seller_package')
          ->where('user_id', $user->id)
          ->latest('id')
          ->first()
      )?->seller_package?->geographic_level > 0;

      if ($requiresPayment) {
        $hasApprovedPayment = SellerPackagePayment::where('user_id', $user->id)
          ->where('approval', 1)
          ->exists();
        if (!$hasApprovedPayment) {
          return response()->json(0);
        }
      }
    }

    $firstTimeApproval = $turnOn && is_null($user->first_approved_at);

    try {
      $user->approval = $turnOn ? 1 : 0;

      // Guard for envs without this column
      if ($firstTimeApproval && Schema::hasColumn('users', 'first_approved_at')) {
        $user->first_approved_at = now();
      }

      $user->save();

      if ($firstTimeApproval && class_exists(\App\Notifications\SellerAccountApprovedNotification::class)) {
        try {
          $user->notify(new \App\Notifications\SellerAccountApprovedNotification($user));
        } catch (\Throwable $e) {
          // ignore notify errors
        }
      }

      // Return the simple 1/0 that your other app expects
      return response()->json(1);
    } catch (\Throwable $e) {
      \Log::error('update_seller_approval failed', [
        'user_id' => $userId,
        'status'  => $turnOn ? 1 : 0,
        'error'   => $e->getMessage(),
      ]);
      return response()->json(0);
    }
  }

  public function update_shop_publish(Request $request)
  {
    $shop = Shop::findOrFail($request->id);
    $shop->published = $request->status;

    cache_clear();

    if ($shop->save()) {
      return 1;
    }
    return 0;
  }

  public function profile_modal(Request $request)
  {
    $seller = User::findOrFail($request->id);
    $assignedName = $this->resolveAssignedLocation($seller);

    return view('addon:multivendor::admin.sellers.profile_modal', compact('seller', 'assignedName'));
  }

  private function resolveAssignedLocation(User $seller): string
  {
    $leafId = (int) ($seller->seller_assign_location ?? 0);
    $level  = (int) ($seller->geographic_level ?? 0);

    if ($leafId > 0 && $level > 0) {
      switch ($level) {
        case 1: {
            $b = Barangay::select('id', 'name', 'city_id')->find($leafId);
            if ($b) {
              $c  = City::select('id', 'name', 'state_id')->find($b->city_id);
              $st = $c ? State::select('id', 'name', 'region_id')->find($c->state_id) : null;
              return trim($b->name
                . ($c ? ', ' . $c->name : '')
                . ($st ? ', ' . $st->name : ''));
            }
            break;
          }
        case 2: {
            $c  = City::select('id', 'name', 'state_id')->find($leafId);
            if ($c) {
              $st = State::select('id', 'name', 'region_id')->find($c->state_id);
              return trim($c->name . ($st ? ', ' . $st->name : ''));
            }
            break;
          }
        case 3: {
            $st = State::select('id', 'name', 'region_id')->find($leafId);
            if ($st) {
              $r = Region::select('id', 'name')->find($st->region_id);
              return trim($st->name . ($r ? ', ' . $r->name : ''));
            }
            break;
          }
        case 4: {
            $r = Region::select('id', 'name')->find($leafId);
            if ($r) return $r->name;
            break;
          }
      }
    }

    if ($b = Barangay::select('id', 'name', 'city_id')->where('seller_assign', $seller->id)->first()) {
      $c  = City::select('id', 'name', 'state_id')->find($b->city_id);
      $st = $c ? State::select('id', 'name', 'region_id')->find($c->state_id) : null;
      return trim($b->name
        . ($c ? ', ' . $c->name : '')
        . ($st ? ', ' . $st->name : ''));
    }

    if ($c = City::select('id', 'name', 'state_id')->where('seller_assign', $seller->id)->first()) {
      $st = State::select('id', 'name', 'region_id')->find($c->state_id);
      return trim($c->name . ($st ? ', ' . $st->name : ''));
    }

    if ($st = State::select('id', 'name', 'region_id')->where('seller_assign', $seller->id)->first()) {
      $r = Region::select('id', 'name')->find($st->region_id);
      return trim($st->name . ($r ? ', ' . $r->name : ''));
    }

    if ($r = Region::select('id', 'name')->where('seller_assign', (string) $seller->id)->first()) {
      return $r->name;
    }

    return '—';
  }

  public function payment_modal(Request $request)
  {
    $seller = User::findOrFail($request->id);
    return view('addon:multivendor::admin.seller_payouts.payment_modal', compact('seller'));
  }

  public function seller_products(Request $request)
  {
    $col_name = null;
    $query = null;
    $sort_search = null;
    $user_id = null;

    $admin = User::where('user_type', 'admin')->first();
    $products = Product::orderBy('created_at', 'desc')->where('user_id', '!=', $admin->user->id);

    if ($request->user_id != null) {
      $user_id = $request->user_id;
      $products = $products->where('user_id', $user_id);
    }

    if ($request->search != null) {
      $products = $products->where('name', 'like', '%' . $request->search . '%');
      $sort_search = $request->search;
    }
    if ($request->type != null) {
      $var = explode(",", $request->type);
      $col_name = $var[0];
      $query = $var[1];
      $products = $products->orderBy($col_name, $query);
      $sort_type = $request->type;
    }

    $products = $products->paginate(15);
    $type = 'All';

    return view('addon:multivendor::admin.seller_products', compact('products', 'type', 'col_name', 'query', 'sort_search', 'user_id'));
  }

  public function seller_orders(Request $request)
  {
    $payment_status = null;
    $delivery_status = null;
    $sort_search = null;

    $admin = User::where('user_type', 'admin')->first();
    $orders = Order::with(['combined_order', 'user'])
      ->whereNotNull('code');

    if ($request->has('search') && $request->search != null) {
      $sort_search = $request->search;
      $orders = $orders->whereHas('combined_order', function ($query) use ($sort_search) {
        $query->where('code', 'like', '%' . $sort_search . '%');
      });
    }

    if ($request->payment_status != null) {
      $orders = $orders->where('payment_status', $request->payment_status);
      $payment_status = $request->payment_status;
    }

    if ($request->delivery_status != null) {
      $orders = $orders->where('delivery_status', $request->delivery_status);
      $delivery_status = $request->delivery_status;
    }

    $orders = $orders->latest()->paginate(15);
    return view('addon:multivendor::admin.orders.seller_orders', compact('orders', 'payment_status', 'delivery_status', 'sort_search'));
  }

  public function seller_order_show($id)
  {
    $order = Order::with(['orderDetails.product', 'orderDetails.variation.combinations', 'order_updates.user'])->findOrFail($id);

    $userId = $order->user_id;
    $user   = User::findOrFail($userId);
    $discount = $user->seller_package->prod_discount;

    return view('backend.orders.seller_orders_show', compact('order', 'discount'));
  }
}
