<?php

namespace App\Http\Controllers\Merchant;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\Product;
use App\Models\CommissionHistory;

class DashboardController extends Controller
{
  public function index()
  {
    $uid = auth()->id();

    // Canonical product scope for this merchant:
    // primary: products.merchant_id = $uid
    // fallback: legacy records where products.user_id = $uid
    $myProductIds = Product::query()
      ->where(function ($q) use ($uid) {
        $q->where('merchant_id', $uid)
          ->orWhere('user_id', $uid);
      })
      ->pluck('id');

    // Totals
    $totalProducts = $myProductIds->count();

    // Sum only non-cancelled orders touching my products
    $grossSales = OrderDetail::query()
      ->whereIn('product_id', $myProductIds)
      ->whereHas('order', fn($q) => $q->where('delivery_status', '!=', 'cancelled'))
      ->sum('total');

    // Distinct orders that include at least one of my products
    $orderIdsForMe = OrderDetail::query()
      ->whereIn('product_id', $myProductIds)
      ->pluck('order_id')
      ->unique()
      ->values();

    $totalOrders = $orderIdsForMe->isEmpty()
      ? 0
      : Order::whereIn('id', $orderIdsForMe)->count();

    // Earnings (all-time) — if your ledger records merchant earnings under the merchant user_id
    $earnings = CommissionHistory::where('user_id', $uid)->sum('seller_earning');

    // Normalize status buckets (store lowercases in DB or compare case-insensitively)
    $statusMap = [
      'placed'            => ['order_placed', 'order placed', 'placed'],
      'packed'            => ['order_packed', 'order packed', 'packed'],
      'shipped'           => ['shipped_to_courier', 'shipped to courier'],
      'in_transit'        => ['in_transit', 'in transit'],
      'out_for_delivery'  => ['out_for_delivery', 'out for delivery'],
      'delivered'         => ['delivered'],
    ];

    // Helper: case-insensitive filter
    $statusFilter = function ($statuses) {
      return function ($q) use ($statuses) {
        $q->where(function ($qq) use ($statuses) {
          foreach ($statuses as $s) {
            $qq->orWhereRaw('LOWER(delivery_status) = ?', [mb_strtolower($s)]);
          }
        });
      };
    };

    // Counts per status, scoped to my orders
    if ($orderIdsForMe->isEmpty()) {
      $countPlaced = $countPacked = $countShipped = $countInTransit = $countOutForDelivery = $countDelivered = 0;
    } else {
      $countPlaced         = Order::whereIn('id', $orderIdsForMe)->where($statusFilter($statusMap['placed']))->count();
      $countPacked         = Order::whereIn('id', $orderIdsForMe)->where($statusFilter($statusMap['packed']))->count();
      $countShipped        = Order::whereIn('id', $orderIdsForMe)->where($statusFilter($statusMap['shipped']))->count();
      $countInTransit      = Order::whereIn('id', $orderIdsForMe)->where($statusFilter($statusMap['in_transit']))->count();
      $countOutForDelivery = Order::whereIn('id', $orderIdsForMe)->where($statusFilter($statusMap['out_for_delivery']))->count();
      $countDelivered      = Order::whereIn('id', $orderIdsForMe)->where($statusFilter($statusMap['delivered']))->count();
    }

    // Top products for this merchant
    $topProducts = $myProductIds->isEmpty()
      ? collect()
      : Product::query()
      ->whereIn('products.id', $myProductIds)
      ->where('products.published', 1)
      ->where('products.approved', 1)
      ->select('products.*')
      ->selectSub(function ($q) {
        $q->from('order_details')
          ->join('orders', 'orders.id', '=', 'order_details.order_id')
          ->whereColumn('order_details.product_id', 'products.id')
          // paid policy
          ->whereRaw('LOWER(orders.payment_status) = ?', ['paid'])
          // exclude cancelled
          ->whereRaw("(orders.delivery_status IS NULL OR LOWER(orders.delivery_status) <> 'cancelled')")
          ->selectRaw('COALESCE(SUM(order_details.quantity), 0)');
      }, 'sold_qty')
      ->orderByDesc('sold_qty')
      ->limit(10)
      ->get();

    return view('backend.merchants.dashboard', compact(
      'totalProducts',
      'grossSales',
      'totalOrders',
      'earnings',
      'topProducts',
      'countPlaced',
      'countPacked',
      'countShipped',
      'countInTransit',
      'countOutForDelivery',
      'countDelivered'
    ));
  }
}
