<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Order;
use App\Models\OrderUpdate;
use CoreComponentRepository;
use Illuminate\Http\Request;
use App\Models\SellerPackage;
use App\Models\CommissionHistory;
use App\Jobs\ProcessOrderCommission;
use App\Jobs\SendOrderNotifications;
use Illuminate\Support\Facades\Bus;

class OrderController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_orders'])->only('index');
    $this->middleware(['permission:view_orders'])->only('show');
    $this->middleware(['permission:delete_orders'])->only('destroy');
  }

  public function index(Request $request)
  {
    CoreComponentRepository::instantiateShopRepository();

    $payment_status = null;
    $delivery_status = null;
    $sort_search = null;

    $orders = Order::whereNull('code'); // Exclude POS orders

    if ($request->filled('search')) {
      $sort_search = trim($request->search);

      $orders = $orders->where(function ($q) use ($sort_search) {
        $q->where('name', 'like', "%{$sort_search}%")
          ->orWhere('code', 'like', "%{$sort_search}%")
          ->orWhereHas('combined_order', function ($sub) use ($sort_search) {
            $sub->where('code', 'like', "%{$sort_search}%");
          });
      });
    }

    if ($request->payment_status != null) {
      $orders = $orders->where('payment_status', $request->payment_status);
      $payment_status = $request->payment_status;
    }

    if ($request->delivery_status != null) {
      $orders = $orders->where('delivery_status', $request->delivery_status);
      $delivery_status = $request->delivery_status;
    }

    $orders = $orders->latest()->paginate(15);
    return view('backend.orders.index', compact('orders', 'payment_status', 'delivery_status', 'sort_search'));
  }

  public function show($id)
  {
    $order = Order::with(['orderDetails.product'])->findOrFail($id);

    return view('backend.orders.show', compact('order'));
  }

  public function destroy($id)
  {
    $order = Order::findOrFail($id);
    if ($order != null) {
      foreach ($order->orderDetails as $key => $orderDetail) {
        $orderDetail->delete();
      }
      foreach ($order->refundRequests as $key => $refundRequest) {
        foreach ($refundRequest->refundRequestItems as $key => $refundRequestItem) {
          $refundRequestItem->delete();
        }
        $refundRequest->delete();
      }

      $order_count = Order::where('combined_order_id', $order->combined_order_id)->count();
      if ($order_count == 1) {
        $order->combined_order->delete();
      }
      $order->delete();
      flash(translate('Order has been deleted successfully'))->success();
    } else {
      flash(translate('Something went wrong'))->error();
    }
    return back();
  }

  public function update_delivery_status(Request $request)
  {
    $order = Order::findOrFail($request->order_id);

    if ($order->delivery_status === 'delivered') {
      return response()->json(['error' => 'Cannot update. Order already delivered.'], 403);
    }

    OrderUpdate::create([
      'order_id' => $order->id,
      'user_id' => auth()->user()->id,
      'note' => 'Order status updated to ' . $request->status . '.',
    ]);

    if ($order->delivery_status != 'cancelled' && $request->status == 'cancelled') {
      foreach ($order->orderDetails as $orderDetail) {
        try {
          foreach ($orderDetail->product->categories as $category) {
            $category->sales_amount -= $orderDetail->total;
            $category->save();
          }

          $brand = $orderDetail->product->brand;
          if ($brand) {
            $brand->sales_amount -= $orderDetail->total;
            $brand->save();
          }
        } catch (\Exception $e) {
        }
      }
    }

    $order->delivery_status = $request->status;
    $order->save();

    flash(translate('Delivery status has been updated.'))->success();
    return 1;
  }

  public function add_tracking_information(Request $request)
  {

    $order = Order::findOrFail($request->order_id);

    if ($order->courier_name) {
      $update_note = 'Courier information updated';
    } else {
      $update_note = 'Courier information added';
    }

    $order->courier_name = $request->courier_name;
    $order->tracking_number = $request->tracking_number;
    $order->tracking_url = $request->tracking_url;
    $order->save();

    OrderUpdate::create([
      'order_id' => $order->id,
      'user_id' => auth()->user()->id,
      'note' => $update_note,
    ]);

    flash(translate('Courier information has been updated.'))->success();

    return back();
  }

  public function add_shipping_cost(Request $request)
  {

    $order = Order::findOrFail($request->order_id);

    $update_note = 'Shipping cost updated';

    $order->shipping_cost = $request->shipping_cost_set;

    $order->save();

    OrderUpdate::create([
      'order_id' => $order->id,
      'user_id' => auth()->user()->id,
      'note' => $update_note,
    ]);

    flash(translate('Shipping cost has been added.'))->success();

    return back();
  }

  public function update_payment_status(Request $request)
  {
    // Eager load user + orderDetails for downstream use / jobs
    $order = Order::with(['user', 'orderDetails'])->findOrFail($request->order_id);

    if ($order->shipping_cost == 0) {
      flash(translate('Please set the shipping cost first.'))->error();
      return 1;
    }

    if ($order->payment_status !== 'unpaid') {
      flash(translate('Paid status cannot be changed.'))->error();
      return 1;
    }

    try {
      $order->payment_status = $request->status;
      $order->save();

      // Queue heavy work so the request returns fast
      if ($request->status === 'paid' && !in_array(optional($order->user)->user_type, ['admin', 'staff'], true)) {
        dispatch(new ProcessOrderCommission($order));
        dispatch(new SendOrderNotifications($order));
      }

      // Log update note
      OrderUpdate::create([
        'order_id' => $order->id,
        'user_id'  => auth()->id(),
        'note'     => 'Payment status updated to ' . $request->status . '.',
      ]);

      flash(translate('Payment status has been updated.'))->success();
    } catch (\Throwable $e) {
      \Log::error('Error processing payment status update: ' . $e->getMessage(), [
        'order_id' => $order->id ?? null,
      ]);
      flash(translate('An error occurred while processing the payment status update.'))->error();

      // Best effort rollback
      $order->update(['payment_status' => 'unpaid']);
      return 1;
    }

    return 1;
  }

  // private function reseller_commision($order)
  // {

  //   $user = $order->user;

  //   $seller_package = \App\Models\SellerPackage::find($user->package_id);

  //   $order_price = $order->grand_total - $order->shipping_cost;

  //   //Per qty flat or percentage
  //   if ($seller_package->commission_type === 'percent') {
  //     $seller_earning = ($seller_package->commission * $order_price) / 100;
  //   } else {
  //     $totalQuantity = $order->orderDetails->sum('quantity');
  //     $seller_earning = $totalQuantity * $seller_package->commission;
  //   }

  //   $user->current_balance += $seller_earning;

  //   $commission = new CommissionHistory();
  //   $commission->order_id = $order->id;
  //   $commission->user_id = $user->id;
  //   $commission->seller_earning = $seller_earning;
  //   $commission->details = 'Sale Commission from Order - ' . $order->name;
  //   $commission->save();
  //   $order->save();
  //   $user->save();
  // }
  // private function calculateAndCreateAreaSaleBonus(Order $order)
  // {
  //   if ($order->user && $order->user->referred_by) {
  //     $referrerToCheck = $order->user->referred_by;
  //     $this->processReferralSaleBonus($order->user->referred_by, $order);
  //   }

  //   // CITY LEVEL BONUS
  //   if ($order->city && $order->city->seller_assign) {
  //     $this->processAreaSaleBonus($order->city->seller_assign, $order);
  //   }

  //   // STATE LEVEL BONUS
  //   if ($order->state && $order->state->seller_assign) {
  //     $this->processAreaSaleBonus($order->state->seller_assign, $order);
  //   }

  //   // REGION LEVEL BONUS
  //   if ($order->region && $order->region->seller_assign) {
  //     $this->processAreaSaleBonus($order->region->seller_assign, $order);
  //   }
  // }
  // private function processReferralSaleBonus($referrerToCheck, Order $order)
  // {
  //   $referrerUser = User::find($referrerToCheck);

  //   if (!$referrerUser || !$referrerUser->package_id) {
  //     return;
  //   }

  //   $sellerPackage = SellerPackage::find($referrerUser->package_id);

  //   if ($sellerPackage->ref_sale_commission_type === 'percent') {
  //     $orderSubtotal = $order->orderDetails->sum('total');
  //     $sellerEarning = ($orderSubtotal * $sellerPackage->ref_sale_commission) / 100;
  //   } else {
  //     $totalQuantity = $order->orderDetails->sum('quantity');
  //     $sellerEarning = $totalQuantity * $sellerPackage->ref_sale_commission;
  //   }

  //   $this->createCommissionHistory($referrerToCheck, $sellerEarning, 'Referral Sale Commission.');
  // }
  // private function processAreaSaleBonus($sellerAssign, Order $order)
  // {
  //   $user = User::find($sellerAssign);

  //   if (!$user || $user->approval != 1) {
  //     return;
  //   }

  //   $sellerPackage = SellerPackage::find($user->package_id);

  //   if ($sellerPackage->area_sale_bonus_type === 'percent') {
  //     $orderSubtotal = $order->orderDetails->sum('total');

  //     $sellerEarning = ($orderSubtotal * $sellerPackage->area_sale_bonus) / 100;
  //   } else {
  //     $totalQuantity = $order->orderDetails->sum('quantity');

  //     $sellerEarning = ($totalQuantity * $sellerPackage->area_sale_bonus);
  //   }

  //   $this->createCommissionHistory($sellerAssign, $sellerEarning, 'Area Sale Bonus.');
  // }
  // private function createCommissionHistory($userId, $sellerEarning, $details)
  // {
  //   $commissionHistory = new CommissionHistory();
  //   $commissionHistory->user_id = $userId;
  //   $commissionHistory->seller_earning = $sellerEarning;
  //   $commissionHistory->details = $details;
  //   $commissionHistory->save();

  //   // Retrieve the user instance
  //   $user = User::findOrFail($userId);

  //   // Update the user's current balance
  //   $user->current_balance += $sellerEarning;
  //   $user->save();
  // }
}
