<?php

namespace App\Jobs;

use App\Models\Order;
use App\Models\CommissionHistory;
use App\Models\SellerPackage;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;

class ProcessOrderCommission implements ShouldQueue
{
  use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

  public Order $order;

  public function __construct(Order $order)
  {
    // Load relations needed for commission math
    $this->order = $order->load([
      'user',
      'orderDetails.product',
      'city',
      'province',
      'region',
    ]);
  }

  public function handle(): void
  {
    // After queue deserialization, relations are not preserved; reload them now.
    $this->order->refresh()->loadMissing([
      'user',
      'orderDetails.product',
      'city',
      'province',
      'region',
    ]);

    $order = $this->order;

    // Web orders only (no POS)
    if ($this->shouldSkipCommission($order)) {
      return;
    }

    // 1) Main reseller commission for order owner
    $this->resellerCommission($order);

    // 2) Referral + area bonuses (city/province/region assignees)
    $this->calculateAndCreateAreaSaleBonus($order);

    // No merchant / admin commission here anymore
  }

  // ---------- Helpers for product override math ----------

  private function productOverrideActive(?\App\Models\Product $p): bool
  {
    if (!$p || !$p->is_commission_override_active) return false;
    $now = now();
    if ($p->commission_effective_from && $now->lt($p->commission_effective_from)) return false;
    if ($p->commission_effective_to && $now->gt($p->commission_effective_to)) return false;
    return true;
  }

  private function applyPercent(float $base, float $pct): float
  {
    return round($base * ($pct / 100), 2);
  }

  private function applyFlat(int $qty, float $flatPerQty): float
  {
    return round($qty * $flatPerQty, 2);
  }

  // ---------- Seller (reseller) commission ----------

  private function resellerCommission(Order $order): void
  {
    $user = $order->user;
    if (!$user || !$user->package_id) return;

    $pkg = SellerPackage::find($user->package_id);
    if (!$pkg) return;

    $total = 0.00;

    foreach ($order->orderDetails as $line) {
      $qty  = (int) ($line->quantity ?? 0);
      $base = (float) ($line->total ?? 0);
      $p    = $line->product;

      // Per-product overrides
      if ($this->productOverrideActive($p) && $p->reseller_type && $p->reseller_value !== null) {
        $total += ($p->reseller_type === 'percent')
          ? $this->applyPercent($base, (float) $p->reseller_value)
          : $this->applyFlat($qty, (float) $p->reseller_value);
        continue;
      }

      // Fallback to package settings
      $total += ($pkg->commission_type === 'percent')
        ? $this->applyPercent($base, (float) $pkg->commission)
        : $this->applyFlat($qty, (float) $pkg->commission);
    }

    $sellerEarning = round($total, 2);

    $entry = CommissionHistory::firstOrCreate(
      [
        'order_id'    => $order->id,
        'user_id'     => $user->id,
        'source_type' => 'order_sale',
      ],
      [
        'seller_earning' => $sellerEarning,
        'details'        => 'Sale Commission from Order - ' . ($order->name ?? ('#' . $order->id)),
      ]
    );

    if ($entry->wasRecentlyCreated) {
      $user->increment('current_balance', $sellerEarning);
    }
  }

  // ---------- Referral + Area Bonuses orchestration ----------

  private function calculateAndCreateAreaSaleBonus(Order $order): void
  {
    // Referral bonus for the upline of this buyer
    if ($order->user && $order->user->referred_by) {
      $this->processReferralSaleBonus((int) $order->user->referred_by, $order);
    }

    // CITY LEVEL BONUS
    if ($order->city && $order->city->seller_assign) {
      $this->processAreaSaleBonus((int) $order->city->seller_assign, $order);
    }

    // PROVINCE/STATE LEVEL BONUS
    if ($order->province && $order->province->seller_assign) {
      $this->processAreaSaleBonus((int) $order->province->seller_assign, $order);
    }

    // REGION LEVEL BONUS
    if ($order->region && $order->region->seller_assign) {
      $this->processAreaSaleBonus((int) $order->region->seller_assign, $order);
    }
  }

  private function processReferralSaleBonus(int $referrerUserId, Order $order): void
  {
    $ref = User::find($referrerUserId);
    if (!$ref || !$ref->package_id) return;

    $pkg = SellerPackage::find($ref->package_id);
    if (!$pkg) return;

    $total = 0.00;

    foreach ($order->orderDetails as $line) {
      $qty  = (int) ($line->quantity ?? 0);
      $base = (float) ($line->total ?? 0);
      $p    = $line->product;

      // Per-product override
      if ($this->productOverrideActive($p) && $p->referral_type && $p->referral_value !== null) {
        $total += ($p->referral_type === 'percent')
          ? $this->applyPercent($base, (float) $p->referral_value)
          : $this->applyFlat($qty, (float) $p->referral_value);
        continue;
      }

      // Fallback to package ref_sale settings
      $total += ($pkg->ref_sale_commission_type === 'percent')
        ? $this->applyPercent($base, (float) $pkg->ref_sale_commission)
        : $this->applyFlat($qty, (float) $pkg->ref_sale_commission);
    }

    $sellerEarning = round($total, 2);

    $this->createCommissionHistory(
      userId: $referrerUserId,
      amount: $sellerEarning,
      details: 'Referral Sale Commission.',
      sourceType: 'referral_sale',
      orderId: $order->id
    );
  }

  private function processAreaSaleBonus(int $assigneeUserId, Order $order): void
  {
    $assignee = User::find($assigneeUserId);
    if (!$assignee || (int) $assignee->approval !== 1) return;

    $pkg = SellerPackage::find($assignee->package_id);
    if (!$pkg) return;

    $total = 0.00;

    foreach ($order->orderDetails as $line) {
      $qty  = (int) ($line->quantity ?? 0);
      $base = (float) ($line->total ?? 0);
      $p    = $line->product;

      // Per-product override
      if ($this->productOverrideActive($p) && $p->area_type && $p->area_value !== null) {
        $total += ($p->area_type === 'percent')
          ? $this->applyPercent($base, (float) $p->area_value)
          : $this->applyFlat($qty, (float) $p->area_value);
        continue;
      }

      // Fallback to package area_sale settings
      $total += ($pkg->area_sale_bonus_type === 'percent')
        ? $this->applyPercent($base, (float) $pkg->area_sale_bonus)
        : $this->applyFlat($qty, (float) $pkg->area_sale_bonus);
    }

    $sellerEarning = round($total, 2);

    $this->createCommissionHistory(
      userId: $assigneeUserId,
      amount: $sellerEarning,
      details: 'Area Sale Bonus.',
      sourceType: 'area_bonus',
      orderId: $order->id
    );
  }

  // ---------- Shared utilities ----------

  private function createCommissionHistory(
    int $userId,
    float $amount,
    string $details,
    string $sourceType,
    ?int $orderId = null
  ): void {
    $entry = CommissionHistory::firstOrCreate(
      [
        'order_id'    => $orderId,
        'user_id'     => $userId,
        'source_type' => $sourceType,
      ],
      [
        'seller_earning' => round($amount, 2),
        'details'        => $details,
      ]
    );

    if ($entry->wasRecentlyCreated) {
      DB::table('users')
        ->where('id', $userId)
        ->update([
          'current_balance' => DB::raw('COALESCE(current_balance,0) + ' . round($amount, 2)),
        ]);
    }
  }

  private function shouldSkipCommission(Order $order): bool
  {
    // Web orders only (POS excluded)
    return !Order::where('id', $order->id)->whereNull('code')->exists();
  }
}
