<?php

namespace App\Addons\Multivendor\Http\Controllers\Admin;

use App\Models\City;
use App\Models\User;
use App\Models\State;
use App\Models\Region;
use App\Models\Barangay;
use App\Models\UserLocationAssignment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use App\Models\SellerPackage;
use App\Models\CommissionHistory;
use App\Http\Controllers\Controller;
use App\Models\SellerPackagePayment;
use App\Models\SellerPackageTranslation;
use App\Notifications\SellerReferralNewUserPaymentUploadNotification;
use App\Notifications\SellerReferralOldUserPaymentUploadNotification;

class SellerPackageController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_seller_packages'])->only('index');
    $this->middleware(['permission:show_seller_package_payments'])->only('package_purchase_history');
  }

  public function index()
  {
    $seller_packages = SellerPackage::all();
    return view('addon:multivendor::admin.seller_packages.index', compact('seller_packages'));
  }

  public function create()
  {
    return view('addon:multivendor::admin.seller_packages.create');
  }

  public function store(Request $request)
  {
    $seller_package = new SellerPackage;
    $seller_package->name = $request->name;
    $seller_package->geographic_level = $request->geographic_level;

    $seller_package->commission = $request->commission;
    $seller_package->commission_type = $request->commission_type;

    $seller_package->ref_commission = $request->ref_commission;
    $seller_package->ref_commission_type = $request->ref_commission_type;

    $seller_package->prod_discount = $request->prod_discount;
    $seller_package->prod_discount_type = $request->prod_discount_type;

    $seller_package->ref_sale_commission = $request->ref_sale_commission;
    $seller_package->ref_sale_commission_type = $request->ref_sale_commission_type;

    $seller_package->area_sale_bonus = $request->area_sale_bonus;
    $seller_package->area_sale_bonus_type = $request->area_sale_bonus_type;

    $seller_package->logo = $request->logo;

    if ($seller_package->save()) {
      $seller_package_translation = SellerPackageTranslation::firstOrNew([
        'lang'               => env('DEFAULT_LANGUAGE'),
        'seller_package_id'  => $seller_package->id,
      ]);
      $seller_package_translation->name = $request->name;
      $seller_package_translation->save();

      flash(translate('Package has been added successfully'))->success();
      return redirect()->route('admin.seller_packages.index');
    }

    flash(translate('Something went wrong'))->error();
    return back();
  }

  public function show($id)
  {
    //
  }

  public function edit(Request $request, $id)
  {
    $lang = $request->lang;
    $seller_package = SellerPackage::findOrFail($id);
    return view('addon:multivendor::admin.seller_packages.edit', compact('seller_package', 'lang'));
  }

  public function update(Request $request, $id)
  {
    $seller_package = SellerPackage::findOrFail($id);
    $seller_package->name = $request->name;
    $seller_package->geographic_level = $request->geographic_level;
    $seller_package->amount = $request->amount;
    $seller_package->commission = $request->commission;
    $seller_package->commission_type = $request->commission_type;
    $seller_package->ref_commission = $request->ref_commission;
    $seller_package->ref_commission_type = $request->ref_commission_type;
    $seller_package->prod_discount = $request->prod_discount;
    $seller_package->prod_discount_type = $request->prod_discount_type;
    $seller_package->ref_sale_commission = $request->ref_sale_commission;
    $seller_package->ref_sale_commission_type = $request->ref_sale_commission_type;
    $seller_package->area_sale_bonus = $request->area_sale_bonus;
    $seller_package->area_sale_bonus_type = $request->area_sale_bonus_type;
    $seller_package->logo = $request->logo;

    if ($seller_package->save()) {
      flash(translate('Package has been updated successfully'))->success();
      return redirect()->route('admin.seller_packages.index');
    }

    flash(translate('Something went wrong'))->error();
    return back();
  }

  public function destroy($id)
  {
    $seller_package = SellerPackage::findOrFail($id);
    foreach ($seller_package->seller_package_translations as $seller_package_translation) {
      $seller_package_translation->delete();
    }
    SellerPackage::destroy($id);
    flash(translate('Package has been deleted successfully'))->success();
    return redirect()->route('admin.seller_packages.index');
  }

  public function package_purchase_history()
  {
    $package_payments = SellerPackagePayment::with('seller_package')->latest()->paginate(20);
    return view('addon:multivendor::admin.seller_packages.payment_history', compact('package_payments'));
  }

  public function package_payment_modal(Request $request)
  {
    $seller_package_payment = SellerPackagePayment::with([
      'user',
      'seller_package',
      'requestedRegion',
      'requestedState',
      'requestedCity',
      'requestedBarangay',
      'assignedRegion',
      'assignedState',
      'assignedCity',
      'assignedBarangay',
    ])->findOrFail($request->id);

    return view('addon:multivendor::admin.seller_packages.package_payment_modal', compact('seller_package_payment'));
  }

  private function labelForLevel(int $level): string
  {
    return match ($level) {
      1 => 'Reseller',          // barangay
      2 => 'City Distributor',  // city
      3 => 'Provincial Dist.',  // state
      4 => 'Regional Dist.',    // region
      default => 'Affiliate',
    };
  }

  public function package_payment_approval(Request $request)
  {
    $payment = SellerPackagePayment::findOrFail($request->id);

    if ((int) $payment->approval === 1) {
      flash(translate('Payment already approved.'))->warning();
      return back();
    }

    $user          = $payment->user;
    $userId        = $user->id;
    $sellerPackage = SellerPackage::findOrFail($payment->seller_package_id);

    // Derive level + leaf id robustly
    [$level, $locId] = $this->deriveAssignment($payment, $sellerPackage);

    try {
      DB::beginTransaction();

      // 0) Free previous assignment when switching level/location
      $this->freePreviousAssignment($user, (int) $level, (int) $locId);

      // 1) Claim or reset assignment
      if ($level > 0 && $locId > 0) {
        switch ($level) {
          case 1: { // barangay
              $geo = Barangay::whereKey($locId)->lockForUpdate()->firstOrFail();
              if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $userId) {
                DB::rollBack();
                flash(translate('This barangay is already assigned to another user.'))->error();
                return back();
              }
              $geo->seller_assign = $userId;
              $geo->save();
              break;
            }
          case 2: { // city
              $geo = City::whereKey($locId)->lockForUpdate()->firstOrFail();
              if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $userId) {
                DB::rollBack();
                flash(translate('This city is already assigned to another user.'))->error();
                return back();
              }
              $geo->seller_assign = $userId;
              $geo->save();
              break;
            }
          case 3: { // state
              $geo = State::whereKey($locId)->lockForUpdate()->firstOrFail();
              if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $userId) {
                DB::rollBack();
                flash(translate('This province is already assigned to another user.'))->error();
                return back();
              }
              $geo->seller_assign = $userId;
              $geo->save();
              break;
            }
          case 4: { // region (seller_assign is VARCHAR)
              $geo = Region::whereKey($locId)->lockForUpdate()->firstOrFail();
              if (!is_null($geo->seller_assign) && (string) $geo->seller_assign !== (string) $userId) {
                DB::rollBack();
                flash(translate('This region is already assigned to another user.'))->error();
                return back();
              }
              $geo->seller_assign = (string) $userId;
              $geo->save();
              break;
            }
        }

        // Persist to users table
        $user->seller_assign_location = $locId;   // bigint on users
        $user->geographic_level       = $level;   // 1..4
        $user->save();

        Cache::forget("user:{$user->id}:assigned_name");

        // Backfill payment’s fields if they were blank
        if (empty($payment->seller_assign_location)) {
          $payment->seller_assign_location = (string) $locId; // varchar on payments
        }
        if (empty($payment->geographic_level)) {
          $payment->geographic_level = $level;
        }
      } else {
        // level == 0 (Affiliate etc.) → no territory
        $user->seller_assign_location = null;
        $user->geographic_level       = 0;
        $user->save();

        Cache::forget("user:{$user->id}:assigned_name");

        // Deactivate any normalized rows
        UserLocationAssignment::where('user_id', $user->id)->update(['is_active' => false]);

        // Normalize payment record too
        $payment->seller_assign_location = null;
        $payment->geographic_level       = 0;
      }

      // 2) Set current package on user
      $user->package_id  = $sellerPackage->id;
      $user->seller_type = $sellerPackage->name;
      $user->save();

      // 3) Normalize assignment table: deactivate old, activate current (levels >= 2)
      if ($level >= 2 && $locId > 0) {
        UserLocationAssignment::where('user_id', $userId)->update(['is_active' => false]);

        UserLocationAssignment::updateOrCreate(
          [
            'user_id'          => $userId,
            'geographic_level' => $level,
            'location_id'      => $locId,
          ],
          [
            'seller_type' => $sellerPackage->name,
            'is_active'   => true,
          ]
        );
      }

      // 4) Approve payment
      $payment->approval = 1;
      $payment->save();

      // 5) Optional: Shop coherence (legacy)
      if ($shop = $user->shop) {
        $shop->commission        = $sellerPackage->commission;
        $shop->seller_package_id = $sellerPackage->id;
        $shop->approval          = 1;
        $shop->save();
      }

      // 6) Referral commission
      $referrerUser    = User::find($user->referred_by);
      $referrerPackage = $referrerUser ? SellerPackage::find($referrerUser->package_id) : null;
      $sellerEarning   = $referrerPackage ? ($sellerPackage->amount * $referrerPackage->ref_commission / 100) : 0;

      if ($referrerUser && $sellerEarning > 0) {
        $ch = new CommissionHistory();
        $ch->user_id        = $referrerUser->id;
        $ch->seller_earning = $sellerEarning;
        $ch->details        = 'Package Payment';
        $ch->source_type    = 'registration';
        $ch->save();

        $referrerUser->current_balance = (float) ($referrerUser->current_balance ?? 0) + $sellerEarning;
        $referrerUser->save();
      }

      // 7) Notifications
      if ((int) $payment->self_purchased === 1) {
        $user->notify(new SellerReferralOldUserPaymentUploadNotification($user->shop));
      } else {
        $user->notify(new SellerReferralNewUserPaymentUploadNotification($user->shop));
      }

      DB::commit();
      flash(translate('Payment approved'))->success();
      return back();
    } catch (\Throwable $e) {
      DB::rollBack();
      \Log::error('Payment approval failed', [
        'payment_id' => $payment->id,
        'error'      => $e->getMessage(),
      ]);
      flash(translate('Unable to approve this payment.'))->error();
      return back();
    }
  }

  /**
   * Free the user's previous assignment if it differs from the new slot.
   * Must be called inside an open DB transaction.
   */
  private function freePreviousAssignment(User $user, int $newLevel, int $newLocId): void
  {
    $prevLevel = (int) ($user->geographic_level ?? 0);
    $prevLocId = (int) ($user->seller_assign_location ?? 0);

    if ($prevLevel <= 0 || $prevLocId <= 0) return;
    if ($prevLevel === $newLevel && $prevLocId === $newLocId) return;

    switch ($prevLevel) {
      case 1: { // barangay
          $b = Barangay::whereKey($prevLocId)->lockForUpdate()->first();
          if ($b && (int) $b->seller_assign === (int) $user->id) {
            $b->seller_assign = null;
            $b->save();
          }
          break;
        }
      case 2: { // city
          $c = City::whereKey($prevLocId)->lockForUpdate()->first();
          if ($c && (int) $c->seller_assign === (int) $user->id) {
            $c->seller_assign = null;
            $c->save();
          }
          break;
        }
      case 3: { // state
          $s = State::whereKey($prevLocId)->lockForUpdate()->first();
          if ($s && (int) $s->seller_assign === (int) $user->id) {
            $s->seller_assign = null;
            $s->save();
          }
          break;
        }
      case 4: { // region (VARCHAR)
          $r = Region::whereKey($prevLocId)->lockForUpdate()->first();
          if ($r && (string) $r->seller_assign === (string) $user->id) {
            $r->seller_assign = null;
            $r->save();
          }
          break;
        }
    }

    // Deactivate any prior normalized rows
    UserLocationAssignment::where('user_id', $user->id)->update(['is_active' => false]);
  }

  /**
   * Resolve level + leaf location id from payment, falling back to package + requested_*.
   */
  private function deriveAssignment(SellerPackagePayment $payment, SellerPackage $package): array
  {
    $level = (int) ($payment->geographic_level ?: $package->geographic_level ?: 0);
    $locId = (int) ($payment->seller_assign_location ?: 0);

    if ($level > 0 && $locId === 0) {
      switch ($level) {
        case 1:
          $locId = (int) ($payment->requested_barangay_id ?: 0);
          break;
        case 2:
          $locId = (int) ($payment->requested_city_id ?: 0);
          break;
        case 3:
          $locId = (int) ($payment->requested_state_id ?: 0);
          break;
        case 4:
          $locId = (int) ($payment->requested_region_id ?: 0);
          break;
      }
    }
    return [$level, $locId];
  }
}
