<?php

namespace App\Addons\Multivendor\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SellerPayout;
use App\Models\User;
use Carbon\Carbon;

class SellerPayoutController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_payouts'])->only('index');
    $this->middleware(['permission:show_payout_requests'])->only('payout_requests', 'payment_modal', 'pay_to_seller');
  }

  /**
   * Paid payouts history (admin)
   */
  public function index(Request $request)
  {
    $user_id   = $request->user_id;
    $date_range = $request->date_range;

    $payouts = SellerPayout::with('user')
      ->where('status', 'paid');

    if (!empty($date_range)) {
      [$from, $to] = array_map('trim', explode(' / ', $date_range));
      $from = Carbon::parse($from)->startOfDay();
      $to   = Carbon::parse($to)->endOfDay();
      $payouts->whereBetween('created_at', [$from, $to]);
    }

    if (!empty($user_id)) {
      $payouts->where('user_id', (int) $user_id);
    }

    $payouts = $payouts->latest()->paginate(20);

    // Note: we now pass user_id (not shop_id)
    return view('addon:multivendor::admin.seller_payouts.payout_history', compact('payouts', 'date_range', 'user_id'));
  }

  /**
   * Payout requests (pending) list (admin)
   */
  public function payout_requests()
  {
    $payout_requests = SellerPayout::with('user')
      ->where('status', 'requested')
      ->latest()
      ->paginate(15);

    return view('addon:multivendor::admin.payout_requests.index', compact('payout_requests'));
  }

  /**
   * Payment modal (admin) – loads data for a specific request
   */
  public function payment_modal(Request $request)
  {
    $seller_withdraw_request = \App\Models\SellerPayout::with('user')->findOrFail($request->id);

    $seller = $seller_withdraw_request->user;   // <- the User (seller)
    if (!$seller) {
      abort(404, 'User not found for this payout request');
    }

    // point to the payout_requests modal path you’re actually using
    return view(
      'addon:multivendor::admin.payout_requests.payment_modal',
      compact('seller', 'seller_withdraw_request')
    );
  }

  /**
   * Pay the seller (admin action)
   */
  public function pay_to_seller(Request $request)
  {
    $user = \App\Models\User::findOrFail($request->user_id);

    if (($user->current_balance ?? 0) < $request->amount) {
      flash(translate('You can not pay more than seller balance'))->error();
      return back();
    }

    $payment = $request->has('withdraw_request_id')
      ? \App\Models\SellerPayout::findOrFail($request->withdraw_request_id)
      : new \App\Models\SellerPayout;

    $payment->user_id        = $user->id;
    $payment->status         = 'paid';
    $payment->paid_amount    = $request->amount;
    $payment->payment_method = $request->payment_option;
    $payment->txn_code       = $request->txn_code;
    $payment->save();

    $user->current_balance = (float) $user->current_balance - (float) $request->amount;
    $user->save();

    flash(translate('Payment completed'))->success();
    return back();
  }
}
