<?php

namespace App\Addons\Multivendor\Http\Controllers\Seller;

use DB;
use Auth;
use Hash;
use Cache;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Order;
use App\Models\Product;
use App\Models\ProductTax;
use App\Models\SellerPackage;
use App\Models\ProductCategory;
use App\Models\ProductAttribute;
use App\Models\ProductVariation;
use App\Models\CommissionHistory;
use App\Models\ProductTranslation;
use App\Models\ManualPaymentMethod;
use App\Models\SellerPackagePayment;
use App\Models\ProductAttributeValue;
use App\Models\ProductVariationCombination;

class SellerController extends Controller
{
  public function seller_dashboard(Request $request)
  {
    $userId = Auth::id();

    // Counts
    $referralCount = User::where('referred_by', $userId)->count();

    // Commissions (breakdown)
    $saleCommision = CommissionHistory::where('user_id', $userId)
      ->where('details', 'Order Payment.')
      ->sum('seller_earning');

    $referralCommision = CommissionHistory::where('user_id', $userId)
      ->where('details', 'Package Payment.')
      ->sum('seller_earning');

    $referralSaleCommision = CommissionHistory::where('user_id', $userId)
      ->where('details', 'Referral Sale Commission.')
      ->sum('seller_earning');

    $areaSaleBonus = CommissionHistory::where('user_id', $userId)
      ->where('details', 'Area Sale Bonus.')
      ->sum('seller_earning');

    // Gross commissions (all-time)
    $grossAllTime = CommissionHistory::where('user_id', $userId)->sum('seller_earning');

    // Deduct only payouts that are PAID
    $paidPayouts = DB::table('seller_payouts')
      ->where('user_id', $userId)
      ->where('status', 'paid')
      ->sum(DB::raw('COALESCE(paid_amount, requested_amount, 0)'));

    // Net earnings = gross commissions − paid payouts
    $allTime = max(0, $grossAllTime - $paidPayouts);

    // Today earnings
    $todayEarnings = CommissionHistory::where('user_id', $userId)
      ->whereDate('created_at', today())
      ->sum('seller_earning');

    // Current balance from users table
    $currentBalance = (float) (Auth::user()->current_balance ?? 0);

    // Latest referrals’ package payments
    $latestPackagePayments = SellerPackagePayment::whereHas('user', function ($query) use ($userId) {
      $query->where('referred_by', $userId);
    })
      ->where('approval', 1)
      ->latest()
      ->limit(8)
      ->get();

    // Base query for this seller's orders on dashboard (exclude POS / special "code" orders)
    $ordersBase = Order::where('user_id', $userId)
      ->whereNull('code');

    // Order status counts
    $orderPlacedCount      = (clone $ordersBase)->where('delivery_status', 'order_placed')->count();
    $orderPackedCount      = (clone $ordersBase)->where('delivery_status', 'order packed')->count();
    $shippedToCourierCount = (clone $ordersBase)->where('delivery_status', 'shipped to courier')->count();
    $inTransitCount        = (clone $ordersBase)->where('delivery_status', 'in transit')->count();
    $outForDeliveryCount   = (clone $ordersBase)->where('delivery_status', 'out for delivery')->count();
    $deliveredCount        = (clone $ordersBase)->where('delivery_status', 'delivered')->count();

    // Recent orders
    $recentOrders = (clone $ordersBase)
      ->latest()
      ->limit(8)
      ->get();

    // Cached chart data (kept for compatibility even if the blade doesn’t use it yet)
    $cached_graph_data = Cache::remember('cached_graph_data-' . $userId, 86400, function () use ($userId) {
      $item = [];
      for ($i = 1; $i <= 12; $i++) {
        $item['sales_number_per_month'][$i] = Order::where('user_id', $userId)
          ->where('delivery_status', '!=', 'cancelled')
          ->whereMonth('created_at', $i)
          ->whereYear('created_at', date('Y'))
          ->count();

        $item['sales_amount_per_month'][$i] = Order::where('user_id', $userId)
          ->where('delivery_status', '!=', 'cancelled')
          ->whereMonth('created_at', $i)
          ->whereYear('created_at', date('Y'))
          ->sum('grand_total');
      }
      return $item;
    });

    return view('addon:multivendor::seller.dashboard', compact(
      'cached_graph_data',
      'referralCount',
      'saleCommision',
      'referralCommision',
      'referralSaleCommision',
      'areaSaleBonus',
      'allTime',               // Net after paid payouts
      'currentBalance',
      'latestPackagePayments',
      'grossAllTime',
      'paidPayouts',
      'todayEarnings',
      'orderPlacedCount',
      'orderPackedCount',
      'shippedToCourierCount',
      'inTransitCount',
      'outForDeliveryCount',
      'deliveredCount',
      'recentOrders'
    ));
  }

  public function profile()
  {
    return view('addon:multivendor::seller.profile');
  }

  public function profile_update(Request $request)
  {
    $user = Auth::user();
    $user->name = $request->name;
    $user->phone = $request->phone;
    if ($request->new_password != null && ($request->new_password == $request->confirm_password)) {
      $user->password = Hash::make($request->new_password);
    }
    $user->avatar = $request->avatar;
    if ($user->save()) {
      flash(translate('Your Profile has been updated successfully!'))->success();
      return back();
    }
    flash(translate('Sorry! Something went wrong.'))->error();
    return back();
  }

  public function seller_products_list(Request $request)
  {
    $search = null;
    $products = Product::orderBy('created_at', 'desc');
    if ($request->has('search')) {
      $search = $request->search;
      $products = $products->where('name', 'like', '%' . $search . '%');
    }
    $products = $products->where('published', 1);
    $products = $products->paginate(10);
    return view('addon:multivendor::seller.products.index', compact('products', 'search'));
  }

  public function seller_product_show($id)
  {
    $product = Product::withCount('reviews', 'wishlists')
      ->with('variations.combinations')
      ->findOrFail($id);

    return view('addon:multivendor::seller.products.show', [
      'product' => $product
    ]);
  }

  public function seller_product_published(Request $request)
  {
    $shop = auth()->user()->shop;
    if (seller_package_validity_check($shop->seller_package, $shop->package_invalid_at) != 'active') {
      return response()->json([
        'success' => false,
        'message' => translate('Please upgrade your package for changing status.')
      ]);
    }

    $product = Product::findOrFail($request->id);
    $product->published = $request->status;
    $product->save();

    cache_clear();

    return response()->json([
      'success' => true,
      'message' => translate('Products status updated successfully')
    ]);
  }

  public function seller_product_duplicate(Request $request, $id)
  {
    $product = Product::findOrFail($id);
    if ($product->shop_id != auth()->user()->shop_id) {
      abort(403);
    }
    $product_new = $product->replicate();
    $product_new->slug = Str::slug($product_new->name, '-') . '-' . strtolower(Str::random(5));

    if ($product_new->save()) {
      foreach ($product->variations as $variation) {
        $p_variation = new ProductVariation;
        $p_variation->product_id = $product_new->id;
        $p_variation->code = $variation->code;
        $p_variation->price = $variation->price;
        $p_variation->stock = $variation->stock;
        $p_variation->sku = $variation->sku;
        $p_variation->img = $variation->img;
        $p_variation->save();

        foreach ($variation->combinations as $combination) {
          $p_variation_comb = new ProductVariationCombination;
          $p_variation_comb->product_id = $product_new->id;
          $p_variation_comb->product_variation_id = $p_variation->id;
          $p_variation_comb->attribute_id = $combination->attribute_id;
          $p_variation_comb->attribute_value_id = $combination->attribute_value_id;
          $p_variation_comb->save();
        }
      }

      foreach ($product->attributes as $attribute) {
        $p_attribute = new ProductAttribute;
        $p_attribute->product_id = $product_new->id;
        $p_attribute->attribute_id = $attribute->attribute_id;
        $p_attribute->save();
      }

      foreach ($product->attribute_values as $attribute_value) {
        $p_attr_value = new ProductAttributeValue;
        $p_attr_value->product_id = $product_new->id;
        $p_attr_value->attribute_id = $attribute_value->attribute_id;
        $p_attr_value->attribute_value_id = $attribute_value->attribute_value_id;
        $p_attr_value->save();
      }

      foreach ($product->product_translations as $translation) {
        $product_translation = new ProductTranslation;
        $product_translation->product_id = $product_new->id;
        $product_translation->name = $translation->name;
        $product_translation->unit = $translation->unit;
        $product_translation->description = $translation->description;
        $product_translation->lang = $translation->lang;
        $product_translation->save();
      }

      foreach ($product->product_categories as $category) {
        $p_category = new ProductCategory;
        $p_category->product_id = $product_new->id;
        $p_category->category_id = $category->category_id;
        $p_category->save();
      }

      foreach ($product->taxes as $tax) {
        $p_tax = new ProductTax;
        $p_tax->product_id = $product_new->id;
        $p_tax->tax_id = $tax->tax_id;
        $p_tax->tax = $tax->tax;
        $p_tax->tax_type = $tax->tax_type;
        $p_tax->save();
      }

      flash(translate('Product has been duplicated successfully'))->success();
      return redirect()->route('seller.products');
    } else {
      flash(translate('Something went wrong'))->error();
      return back();
    }
  }

  public function seller_product_destroy($id)
  {
    $product = Product::findOrFail($id);
    if ($product->shop_id != auth()->user()->shop_id) {
      abort(403);
    }

    $product->product_translations()->delete();
    $product->variations()->delete();
    $product->variation_combinations()->delete();
    $product->reviews()->delete();
    $product->product_categories()->delete();
    $product->offers()->delete();
    $product->wishlists()->delete();
    $product->attributes()->delete();
    $product->attribute_values()->delete();
    $product->taxes()->delete();

    if (Product::destroy($id)) {
      flash(translate('Product has been deleted successfully'))->success();
      return redirect()->route('seller.products');
    } else {
      flash(translate('Something went wrong'))->error();
      return back();
    }
  }

  public function orders(Request $request)
  {
    $payment_status = null;
    $delivery_status = null;
    $sort_search = null;

    $orders = Order::where('user_id', auth()->id())
      ->whereNull('code');

    $user = auth()->user();
    $discount = 0;
    $sellerPackage = SellerPackage::find($user->package_id);

    if ($request->has('search') && $request->search != null) {
      $sort_search = $request->search;
      $orders = $orders->where('code', 'like', '%' . $sort_search . '%');
    }
    if ($request->payment_status != null) {
      $orders = $orders->where('payment_status', $request->payment_status);
      $payment_status = $request->payment_status;
    }
    if ($request->delivery_status != null) {
      $orders = $orders->where('delivery_status', $request->delivery_status);
      $delivery_status = $request->delivery_status;
    }

    $orders = $orders->latest()->paginate(15);
    return view('addon:multivendor::seller.orders.index', compact('orders', 'payment_status', 'delivery_status', 'sort_search'));
  }

  public function referrals(Request $request)
  {
    $referred_shop_id = auth()->id();
    $date_range = $request->get('date_range');

    $referrals = User::with(['sellerPackagePayment.seller_package'])
      ->where('referred_by', $referred_shop_id)
      ->latest()
      ->paginate(10);

    return view(
      'addon:multivendor::seller.referrals.referrals',
      compact('referrals', 'referred_shop_id', 'date_range')
    );
  }

  public function orders_show($id)
  {
    $order = Order::with([
      'orderDetails.product',
      'orderDetails.variation.combinations',
      'commission_histories',
    ])->findOrFail($id);

    if ($order->user_id != auth()->id()) {
      abort(403);
    }

    $discount = optional($order->user->seller_package)->prod_discount;

    $earningHistories = $order->commission_histories()
      ->where('user_id', auth()->id())
      ->whereNotIn('details', ['Admin Commission', 'Merchant Net', 'Order Payment.'])
      ->latest()
      ->get();

    $hasProductOverrides = $order->orderDetails->contains(function ($d) {
      $p = $d->product;
      if (!$p) return false;
      if (!$p->is_commission_override_active) return false;

      $now = now();
      if ($p->commission_effective_from && $now->lt($p->commission_effective_from)) return false;
      if ($p->commission_effective_to && $now->gt($p->commission_effective_to)) return false;

      return (
        ($p->reseller_type && $p->reseller_value !== null) ||
        ($p->referral_type && $p->referral_value !== null) ||
        ($p->area_type && $p->area_value !== null)
      );
    });

    return view(
      'addon:multivendor::seller.orders.show',
      compact('order', 'discount', 'earningHistories', 'hasProductOverrides')
    );
  }

  public function seller_product_reviews(Request $request)
  {
    $reviews = DB::table('reviews')
      ->orderBy('id', 'desc')
      ->join('products', 'reviews.product_id', '=', 'products.id')
      ->where('products.shop_id', Auth::user()->id)
      ->select('reviews.id')
      ->distinct()
      ->paginate(10);

    return view('addon:multivendor::seller.product_reviews', compact('reviews'));
  }

  public function payment_proof_upload(Request $request)
  {
    $user = Auth::user();
    if (!$user) {
      return redirect()->route('login');
    }

    $payment = $user->sellerPackagePayment()->with('seller_package')->first();
    $package = optional($payment)->seller_package
      ?? optional($user->package)
      ?? SellerPackage::find($user->package_id);

    $level = (int) ($package->geographic_level ?? 0);

    if ($level === 0) {
      return redirect()
        ->route('thank-you-for-registration')
        ->with('status', 'Your account is pending approval.');
    }

    if ($payment && (int) $payment->approval === 1) {
      return redirect()
        ->route('login')
        ->with('status', 'Your payment is approved. Your account is pending admin approval.');
    }

    $manual_payment_methods = ManualPaymentMethod::all();
    $packageName  = optional($package)->name;
    $referrerName = optional(User::find($user->referred_by))->name;

    return view('frontend.pages.upload_payment_proof', compact(
      'manual_payment_methods',
      'payment',
      'packageName',
      'referrerName',
      'user'
    ));
  }

  public function payment_upload_modal(Request $request)
  {
    $manual_payment_methods = ManualPaymentMethod::all();
    return view('frontend.pages.payment_upload_modal', compact('manual_payment_methods'));
  }

  public function upload_payment_proof(Request $request)
  {
    $request->validate([
      'payment_method' => 'required|string',
      'photo_proof'    => 'required',
      'payment_id'     => [
        'required',
        'integer',
        Rule::exists('seller_package_payments', 'id')->where(function ($q) {
          $q->where('user_id', auth()->id());
        }),
      ],
    ]);

    $payment = SellerPackagePayment::with('seller_package')
      ->where('id', $request->payment_id)
      ->where('user_id', auth()->id())
      ->firstOrFail();

    if ((int) $payment->approval === 1) {
      flash(translate('Your payment is already approved.'))->info();
      return back();
    }

    $payment->payment_proof_upload  = $request->photo_proof;
    $payment->payment_method        = $request->payment_method;
    $payment->approval              = 3;
    $payment->payment_proof_tran_id = $payment->id
      . '-' . auth()->id()
      . '-' . ($payment->seller_package_id ?? 'pkg')
      . '-' . now()->format('YmdHis');
    $payment->save();

    $user = auth()->user();
    $shop = \App\Models\Shop::where('user_id', $user->id)->first();

    try {
      $admin = User::where('user_type', 'admin')->first();
      if ($admin && $shop) {
        $admin->notify(new \App\Notifications\SellerUploadPaymentNotification($user));
      }
    } catch (\Throwable $e) {
    }

    try {
      if ($shop) {
        $user->notify(new \App\Notifications\SellersUploadedPaymentNotification($user));
      }
    } catch (\Throwable $e) {
    }

    flash(translate('Payment proof uploaded successfully.'))->success();
    return back();
  }

  public function linkCopied()
  {
    env('HOMEPAGE_URL')
      ? flash(translate('Link copied successfully!'))->success()
      : flash(translate('Error: Link was not set.'))->error();

    return back();
  }
}
