<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\OrderDetail;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\StreamedResponse;

class MerchantEarningsController extends Controller
{
    // Commission: subtotal only (price * quantity)
    // Paid-policy: LOWER(orders.payment_status)='paid' OR in (1,'1',true)

    public function index(Request $request)
    {
        $q    = trim($request->get('q', ''));
        $from = $request->date_from;
        $to   = $request->date_to;

        $merchantQuery = User::query()->where('user_type', 'merchant');

        if ($q !== '') {
            $merchantQuery->where(function ($w) use ($q) {
                $w->where('name', 'like', "%{$q}%")
                  ->orWhere('email', 'like', "%{$q}%")
                  ->orWhere('phone', 'like', "%{$q}%");
            });
        }

        $commissionRate = (float) (get_setting('merchant_commission_rate') ?? 10);
        $paidPayoutStatuses = ['paid','approved','completed']; // LOWER(status) values

        // Orders & Gross (paid only) grouped by owner = COALESCE(merchant_id, user_id)
        $ordersAgg = OrderDetail::query()
            ->join('orders', 'orders.id', '=', 'order_details.order_id')
            ->join('products', 'products.id', '=', 'order_details.product_id')
            ->when($from, fn($qq) => $qq->whereDate('orders.created_at', '>=', $from))
            ->when($to,   fn($qq) => $qq->whereDate('orders.created_at', '<=', $to))
            ->where(function ($qq) {
                $qq->whereRaw('LOWER(orders.payment_status) = ?', ['paid'])
                   ->orWhereIn('orders.payment_status', [1, '1', true]);
            })
            ->selectRaw('COALESCE(products.merchant_id, products.user_id) as merchant_id')
            ->selectRaw('COUNT(DISTINCT orders.id) as orders_count')
            ->selectRaw('SUM(order_details.price * order_details.quantity) as gross_all')
            ->groupBy('merchant_id');

        // Paid payouts per merchant
        $payoutsAgg = DB::table('seller_payouts')
            ->select([
                'user_id',
                DB::raw('SUM(COALESCE(NULLIF(paid_amount,0), requested_amount)) as paid_payouts'),
            ])
            ->whereIn(DB::raw('LOWER(status)'), $paidPayoutStatuses)
            ->groupBy('user_id');

        $merchants = $merchantQuery
            ->leftJoinSub($ordersAgg, 'agg', 'agg.merchant_id', '=', 'users.id')
            ->leftJoinSub($payoutsAgg, 'p', 'p.user_id', '=', 'users.id')
            ->select([
                'users.id',
                'users.name',
                'users.email',
                'users.phone',
                'users.current_balance',
                DB::raw('COALESCE(agg.orders_count,0) as orders_count'),
                DB::raw('COALESCE(agg.gross_all,0) as gross_all'),
                DB::raw('ROUND(COALESCE(agg.gross_all,0) * ' . ($commissionRate / 100) . ', 2) as commission'),
                DB::raw('ROUND(COALESCE(agg.gross_all,0) - COALESCE(agg.gross_all,0) * ' . ($commissionRate / 100) . ', 2) as merchant_net'),
                DB::raw('COALESCE(p.paid_payouts,0) as paid_payouts'),
            ])
            ->orderByDesc('gross_all')
            ->paginate(20)
            ->appends($request->query());

        return view('backend.merchants.earnings.admin.index', [
            'merchants'       => $merchants,
            'commission_rate' => $commissionRate,
            'date_from'       => $from,
            'date_to'         => $to,
            'q'               => $q,
        ]);
    }

    public function show(\App\Models\User $merchant, \Illuminate\Http\Request $request)
{
    abort_unless($merchant->user_type === 'merchant', 404);

    $from = $request->date_from;
    $to   = $request->date_to;

    // Paid policy: 'paid' (case-insensitive) OR truthy(1/'1'/true)
    $paidFilter = function ($q) {
        $q->where(function ($w) {
            $w->whereRaw('LOWER(orders.payment_status) = ?', ['paid'])
              ->orWhereIn('orders.payment_status', [1, '1', true]);
        });
    };

    // Per-order gross for THIS merchant (subtotal only)
    $grossByOrder = \App\Models\OrderDetail::query()
        ->join('orders', 'orders.id', '=', 'order_details.order_id')
        ->join('products', 'products.id', '=', 'order_details.product_id')
        ->whereRaw('COALESCE(products.merchant_id, products.user_id) = ?', [$merchant->id])
        ->when($from, fn($qq) => $qq->whereDate('orders.created_at', '>=', $from))
        ->when($to,   fn($qq) => $qq->whereDate('orders.created_at', '<=', $to))
        ->where(function ($qq) use ($paidFilter) { $paidFilter($qq); })
        ->groupBy('orders.id', 'orders.code', 'orders.created_at')
        ->selectRaw('orders.id AS order_id, orders.code AS order_code, orders.created_at AS created_at, SUM(order_details.price * order_details.quantity) AS gross');

    // Ledger net per order for THIS merchant (immutable)
    $ledgerNet = \App\Models\CommissionHistory::query()
        ->where('user_id', $merchant->id)
        ->where('source_type', 'merchant_sale')
        ->groupBy('order_id')
        ->selectRaw('order_id, SUM(seller_earning) AS merchant_net');

    // Details table
    $details = \DB::query()
        ->fromSub($grossByOrder, 'g')
        ->leftJoinSub($ledgerNet, 'ch', 'ch.order_id', '=', 'g.order_id')
        ->select([
            'g.order_id',
            'g.order_code',
            'g.created_at',
            \DB::raw('g.gross AS gross'),
            // display-only commission
            \DB::raw('(g.gross * (' . ((float)(get_setting('merchant_commission_rate') ?? 10) / 100) . ')) AS commission'),
            \DB::raw('COALESCE(ch.merchant_net, 0) AS merchant_net'),
        ])
        ->orderByDesc('g.created_at')
        ->paginate(25)
        ->appends($request->query());

    // Totals
    $gross_total = (float) \DB::query()
        ->fromSub($grossByOrder, 'g')
        ->sum('g.gross');

    $commission_rate = (float) (get_setting('merchant_commission_rate') ?? 10);
    $admin_commission = round($gross_total * ($commission_rate / 100), 2);

    $merchant_net_total = (float) \App\Models\CommissionHistory::query()
        ->where('user_id', $merchant->id)
        ->where('source_type', 'merchant_sale')
        ->when($from, fn($qq) => $qq->whereDate('created_at', '>=', $from))
        ->when($to,   fn($qq) => $qq->whereDate('created_at', '<=', $to))
        ->sum('seller_earning');

    $paid_payouts = (float) \DB::table('seller_payouts')
        ->where('user_id', $merchant->id)
        ->whereIn(\DB::raw('LOWER(status)'), ['paid', 'approved', 'completed'])
        ->sum(\DB::raw('COALESCE(NULLIF(paid_amount,0), requested_amount)'));

    $current_balance = (float) ($merchant->current_balance ?? 0);

    return view('backend.merchants.earnings.admin.show', [
        'merchant'           => $merchant,
        'details'            => $details,
        'commission_rate'    => $commission_rate,
        'gross_total'        => $gross_total,
        'admin_commission'   => $admin_commission,
        'merchant_net_total' => $merchant_net_total,
        'paid_payouts'       => $paid_payouts,
        'current_balance'    => $current_balance,
        'date_from'          => $from,
        'date_to'            => $to,
    ]);
}

    public function export(Request $request): \Symfony\Component\HttpFoundation\StreamedResponse
{
    $from = $request->date_from;
    $to   = $request->date_to;
    $q    = trim($request->get('q', ''));

    $commissionRate = (float) (get_setting('merchant_commission_rate') ?? 10);
    $rateFactor     = $commissionRate / 100;

    // Paid orders only: 'paid' (case-insensitive) OR truthy(1/'1'/true)
    $paidFilter = function ($qq) {
        $qq->where(function ($w) {
            $w->whereRaw('LOWER(orders.payment_status) = ?', ['paid'])
              ->orWhereIn('orders.payment_status', [1, '1', true]);
        });
    };

    // Aggregate by owner = COALESCE(products.merchant_id, products.user_id)
    $rows = \App\Models\OrderDetail::query()
        ->select([
            DB::raw('COALESCE(products.merchant_id, products.user_id) AS owner_id'),
            DB::raw('COUNT(DISTINCT orders.id) AS orders_count'),
            DB::raw('SUM(order_details.price * order_details.quantity) AS gross_all'),
        ])
        ->join('orders', 'orders.id', '=', 'order_details.order_id')
        ->join('products', 'products.id', '=', 'order_details.product_id')
        ->when($from, fn($qq) => $qq->whereDate('orders.created_at', '>=', $from))
        ->when($to,   fn($qq) => $qq->whereDate('orders.created_at', '<=', $to))
        ->where(function ($qq) use ($paidFilter) { $paidFilter($qq); })
        ->groupBy(DB::raw('COALESCE(products.merchant_id, products.user_id)'));

    // Paid payouts (use paid_amount when present else requested_amount) for statuses paid/approved/completed
    $payouts = DB::table('seller_payouts')
        ->select([
            'user_id',
            DB::raw('SUM(COALESCE(NULLIF(paid_amount,0), requested_amount)) AS paid_payouts'),
        ])
        ->whereIn(DB::raw('LOWER(status)'), ['paid', 'approved', 'completed'])
        ->groupBy('user_id');

    // Base query: merchants with optional search + aggregates
    $data = \App\Models\User::query()
        ->where('user_type', 'merchant')
        ->when($q !== '', function ($w) use ($q) {
            $w->where(function ($x) use ($q) {
                $x->where('name', 'like', "%{$q}%")
                  ->orWhere('email', 'like', "%{$q}%")
                  ->orWhere('phone', 'like', "%{$q}%");
            });
        })
        ->leftJoinSub($rows, 'agg', 'agg.owner_id', '=', 'users.id')
        ->leftJoinSub($payouts, 'p', 'p.user_id', '=', 'users.id')
        ->select([
            'users.id',
            'users.name',
            'users.email',
            'users.phone',
            DB::raw('COALESCE(agg.orders_count,0) AS orders_count'),
            DB::raw('COALESCE(agg.gross_all,0) AS gross_all'),
            DB::raw('COALESCE(p.paid_payouts,0) AS paid_payouts'),
            DB::raw('COALESCE(users.current_balance,0) AS current_balance'),
        ])
        ->orderBy('users.id')
        ->get();

    $filename = 'merchant_earnings_' . now()->format('Ymd_His') . '.csv';

    return response()->streamDownload(function () use ($data, $commissionRate, $rateFactor) {
        $out = fopen('php://output', 'w');
        fputcsv($out, [
            'Merchant ID',
            'Merchant Name',
            'Email',
            'Phone',
            'Orders',
            'Gross',
            'Commission %',
            'Commission',
            'Merchant Net',
            'Paid Payouts',
            'Current Balance',
        ]);

        foreach ($data as $r) {
            $gross       = (float) $r->gross_all;
            $commission  = round($gross * $rateFactor, 2);
            $merchantNet = round($gross - $commission, 2);

            fputcsv($out, [
                $r->id,
                $r->name,
                $r->email,
                $r->phone,
                (int) $r->orders_count,
                number_format($gross, 2, '.', ''),
                $commissionRate,
                number_format($commission, 2, '.', ''),
                number_format($merchantNet, 2, '.', ''),
                number_format((float) $r->paid_payouts, 2, '.', ''),
                number_format((float) $r->current_balance, 2, '.', ''),
            ]);
        }
        fclose($out);
    }, $filename, ['Content-Type' => 'text/csv']);
}

}
