<?php

namespace App\Http\Controllers\Api;

use App\Http\Resources\DownloadedProductResource;
use App\Http\Resources\OrderCollection;
use App\Http\Resources\OrderSingleCollection;
use App\Services\Phone;
use App\Models\Address;
use App\Models\City;
use App\Models\CombinedOrder;
use App\Models\Language;
use App\Models\State;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\OrderUpdate;
use App\Models\Product;
use App\Models\Shop;
use App\Models\Upload;
use App\Models\User;
use App\Notifications\DB\OrderDbNotification;
use App\Notifications\OrderPlacedNotification;
use App\Notifications\SellerInvoiceNotification;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use PDF;

class OrderController extends Controller
{
  public function new_post_order(Request $request)
  {
    if ($request->bearerToken() !== 'aqSm*krXb+!~8SI5O_6V73KaYXB5~F') {
      return response()->json(['error' => 'Unauthorized'], 401);
    }

    \DB::beginTransaction();
    try {
      $customer       = $request->input('customer', []);
      $shipping       = $request->input('shipping_address', []);
      $items          = $request->input('items', []);
      $shipping_cost  = (float) ($request->input('shipping_cost', 0) ?? 0);
      $seller_id      = (int) ($request->input('seller_id') ?? 0);
      $payment_method = $request->input('payment_method');

      // Phone normalize (PH 11-digit 09XXXXXXXXX)
      $normalizedPhone = Phone::normalizePhMobile($customer['phone'] ?? null);
      if (!$normalizedPhone) {
        return response()->json(['error' => 'Invalid Philippine mobile number. Expect 11 digits starting with 09.'], 422);
      }

      // Address resolution
      $province_id  = $shipping['province'] ?? null; // may be ID or state_code
      $stateRec     = \App\Models\State::where('id', $province_id)->orWhere('state_code', $province_id)->first();
      $provinceName = $stateRec?->name ?? (string) $province_id;
      $region_id    = $stateRec?->region_id;

      $cityIn  = $shipping['city'] ?? null;
      $cityRec = $cityIn
        ? (\App\Models\City::where('name', $cityIn)
          ->when($stateRec, fn($q) => $q->where('state_id', $stateRec->id))
          ->first()
          ?? \App\Models\City::where('name', $cityIn)->first())
        : null;
      $city_id   = $cityRec?->id;
      $cityName  = $cityRec?->name ?? (string) $cityIn;

      $brgyIn  = $shipping['barangay'] ?? null;
      $brgyRec = $brgyIn
        ? (\App\Models\Barangay::where('name', $brgyIn)
          ->when($city_id, fn($q) => $q->where('city_id', $city_id))
          ->first()
          ?? \App\Models\Barangay::where('name', $brgyIn)->first())
        : null;
      $barangay_id   = $brgyRec?->id;
      $barangayName  = $brgyRec?->name ?? (string) $brgyIn;

      $address = trim(collect([
        $shipping['street'] ?? '',
        $barangayName,
        $cityName,
        $provinceName,
      ])->filter()->implode(', '));

      // Product resolver (map by woocommerce_id → sku → name; create shadow product if absent)
      $resolveProduct = function (array $it) use ($seller_id) {
        $wooId = (int) ($it['product_id'] ?? 0);
        $sku   = isset($it['sku']) ? trim((string) $it['sku']) : null;
        $name  = isset($it['name']) ? trim((string) $it['name']) : null;

        if ($wooId > 0) {
          if ($p = \App\Models\Product::where('woocommerce_id', $wooId)->first()) return $p;
        }
        if ($sku) {
          if ($p = \App\Models\Product::where('sku', $sku)->first()) return $p;
        }
        if ($name) {
          if ($p = \App\Models\Product::whereRaw('LOWER(name) = ?', [mb_strtolower($name)])
            ->orderByDesc('published')->orderByDesc('id')->first()
          ) return $p;
          $slug = \Illuminate\Support\Str::slug($name, '-');
          if ($p = \App\Models\Product::where('slug', $slug)->orderByDesc('id')->first()) return $p;
        }

        // Create minimal local product using only existing columns
        $qty      = max(1, (int) ($it['quantity'] ?? 1));
        $subtotal = is_numeric($it['subtotal'] ?? null) ? (float) $it['subtotal'] : 0.0;
        $unit     = $qty > 0 && $subtotal > 0 ? ($subtotal / $qty) : 0.0;

        $new = new \App\Models\Product();
        $new->user_id        = $seller_id ?: 1;
        $new->merchant_id    = null;
        $new->woocommerce_id = $wooId ?: null;
        $new->name           = $name ?: ('Woo Product ' . ($wooId ?: 'N/A'));
        $new->slug           = \Illuminate\Support\Str::slug($new->name . '-' . ($wooId ?: \Str::random(5)), '-');
        $new->lowest_price   = $unit;
        $new->highest_price  = $unit;
        $new->sku            = $sku;
        $new->stock          = 1;
        $new->published      = 0;
        $new->min_qty        = 1;
        $new->max_qty        = 0;
        $new->is_variant     = 0;
        $new->save();

        \Log::info('new_post_order: created shadow product', [
          'woo_id' => $wooId,
          'sku' => $sku,
          'name' => $name,
          'product_id' => $new->id
        ]);
        return $new;
      };

      // Build order lines strictly from Woo values
      $lines       = [];
      $items_total = 0.0;

      foreach ($items as $it) {
        $qty = (int) ($it['quantity'] ?? 0);
        if ($qty < 1) {
          \Log::warning('new_post_order: invalid qty, skipping', ['item' => $it]);
          continue;
        }
        $product   = $resolveProduct($it);
        $subtotal  = is_numeric($it['subtotal'] ?? null) ? (float) $it['subtotal'] : ((float) $product->lowest_price * $qty);
        $lineTotal = is_numeric($it['total'] ?? null)    ? (float) $it['total']    : $subtotal;
        $unit      = $qty > 0 ? ($subtotal / $qty) : (float) $product->lowest_price;
        $taxTotal  = is_numeric($it['total_tax'] ?? null) ? (float) $it['total_tax'] : 0.0;

        $items_total += $lineTotal;

        $lines[] = [
          'product'    => $product,
          'qty'        => $qty,
          'unit'       => $unit,
          'tax'        => $taxTotal,
          'line_total' => $lineTotal,
        ];
      }

      if (empty($lines)) {
        \DB::rollBack();
        \Log::warning('new_post_order: no items could be processed', ['items' => $items]);
        return response()->json(['error' => 'No resolvable items'], 422);
      }

      $grand_total = $items_total + $shipping_cost;

      // Combined order
      $combined = new \App\Models\CombinedOrder();
      $combined->code             = now()->format('Ymd-His') . rand(10, 99);
      $combined->shipping_address = $address;
      $combined->billing_address  = $address;
      $combined->grand_total      = $grand_total;
      $combined->save();

      // Order (omit columns not present in your orders table; DO NOT insert billing_address)
      $order = \App\Models\Order::create([
        'user_id'           => $seller_id,
        'combined_order_id' => $combined->id,
        'name'              => trim(($customer['first_name'] ?? '') . ' ' . ($customer['last_name'] ?? '')),
        'email'             => $customer['email'] ?? null,
        'phone'             => $normalizedPhone,
        'shipping_address'  => $address,
        'barangay_id'       => $barangay_id,
        'city_id'           => $city_id,
        'province_id'       => $stateRec?->id,
        'region_id'         => $region_id,
        'grand_total'       => $grand_total,
        'shipping_cost'     => $shipping_cost,
        'delivery_type'     => 'Standard',
        'payment_type'      => $payment_method,
      ]);

      // Order details
      foreach ($lines as $r) {
        \App\Models\OrderDetail::create([
          'order_id'   => $order->id,
          'product_id' => $r['product']->id,
          'price'      => $r['unit'],
          'quantity'   => $r['qty'],
          'tax'        => $r['tax'],
          'total'      => $r['line_total'],
        ]);
      }

      // Seller earning (exclude shipping)
      $seller_commission   = optional(\App\Models\User::find($seller_id))->commission ?? 0;
      $order_price         = $items_total;
      $order->seller_earning = $seller_commission > 0 ? ($seller_commission * $order_price) / 100 : $order_price;
      $order->save();

      // Notifications
      try {
        if (!empty($customer['email'])) {
          \Notification::route('mail', $customer['email'])
            ->notify(new \App\Notifications\OrderPlacedNotification($combined));
        }
        foreach ($combined->orders as $orderRow) {
          $seller = optional($orderRow->orderDetails->first())->product->shop->user ?? null;
          if ($seller && !empty($seller->email)) {
            \Notification::route('mail', $seller->email)->notify(new \App\Notifications\SellerInvoiceNotification($orderRow));
            \Notification::send($seller, new \App\Notifications\SellerInvoiceNotification($orderRow));
            \Notification::send($seller, new \App\Notifications\OrderDbNotification($orderRow));
          }
        }
      } catch (\Throwable $e) {
        \Log::warning('new_post_order notifications', ['err' => $e->getMessage()]);
      }

      \DB::commit();
      return response()->json(['success' => true]);
    } catch (\Throwable $e) {
      \DB::rollBack();
      \Log::error('new_post_order failed', ['err' => $e->getMessage(), 'payload' => $request->all()]);
      return response()->json(['error' => 'Something went wrong.'], 500);
    }
  }


  public function post_store_order(Request $request)
  {
    // Create a combined order
    $combined_order = new CombinedOrder;
    $combined_order->code = date('Ymd-His') . rand(10, 99);

    // Get state details from the request
    $stateCode = $request->billing['state'];
    $state = State::where('state_code', $stateCode)->first();

    // Construct the shipping and billing address
    $shippingAddress = $request->billing['address_1'] . ', ' . $request->billing['address_2'] . ', ' . $request->billing['city'] . ', ' . $state->name;
    $combined_order->shipping_address = $shippingAddress;
    $combined_order->billing_address = $shippingAddress;

    $grand_total = 0;
    $package_number = 1;

    // Process each line item and calculate total
    foreach ($request->line_items as $line_item) {
      $product = Product::find($line_item['product_id']);
      $line_item_total = $product->lowest_price * $line_item['quantity'];
      $grand_total += $line_item_total;
    }

    $delivery_type = $request->shipping_lines[0]['method_title'];
    $shipping_cost = floatval($request->shipping_total);
    $grand_total += $shipping_cost;

    $combined_order->grand_total = $grand_total;
    $combined_order->save();

    // Extract the seller ID from meta_data
    $billing_seller_meta = collect($request->meta_data)->firstWhere('key', '_billing_seller_id');
    $billing_seller_id = $billing_seller_meta['value'];

    // Create the order
    $order = Order::create([
      'shop_id' => $billing_seller_id,
      'combined_order_id' => $combined_order->id,
      'name' => $request->billing['first_name'] . ' ' . $request->billing['last_name'],
      'email' => $request->billing['email'],
      'phone' => $request->billing['phone'],
      'shipping_address' => $combined_order->shipping_address,
      'billing_address' => $combined_order->shipping_address,
      'customer_city' => $request->billing['city'],
      'customer_state' => $state->name,
      'grand_total' => $grand_total,
      'shipping_cost' => $shipping_cost,
      'delivery_type' => $delivery_type,
      'payment_type' => $request->payment_method_title,
    ]);

    // Calculate seller earnings
    $shop_commission = Shop::find($billing_seller_id)->commission;
    $order_price = $order->grand_total - $order->shipping_cost;
    $seller_earning = $shop_commission > 0 ? ($shop_commission * $order_price) / 100 : $order_price;
    $order->seller_earning = $seller_earning;
    $order->save();

    // Create OrderDetail records
    foreach ($request->line_items as $line_item) {
      $product = Product::find($line_item['product_id']);
      $orderDetail = OrderDetail::create([
        'order_id' => $order->id,
        'product_id' => $product->id,
        'price' => $product->lowest_price,
        'quantity' => $line_item['quantity'],
        'total' => $product->lowest_price * $line_item['quantity'],
      ]);
    }

    // Send notifications
    try {
      Notification::route('mail', $request->billing['email'])->notify(new OrderPlacedNotification($combined_order));

      foreach ($combined_order->orders as $order) {
        $seller = $order->orderDetails->first()->product->shop->user;

        // Send Seller Invoice Notification to the shop owner
        Notification::route('mail', $order->shop->user->email)->notify(new SellerInvoiceNotification($order));
        Notification::send($seller, new SellerInvoiceNotification($order));
        Notification::send($seller, new OrderDbNotification($order));
      }
    } catch (\Exception $e) {
      \Log::error('Notification Error: ' . $e->getMessage());
    }

    return response()->json([
      'success' => true,
    ]);
  }

  public function index()
  {
    return new OrderCollection(CombinedOrder::with(['user', 'orders.orderDetails.variation.product', 'orders.orderDetails.variation.combinations', 'orders.shop'])->where('user_id', auth('api')->user()->id)->latest()->paginate(12));
  }

  public function show($order_code)
  {
    $order = CombinedOrder::where('code', $order_code)->with(['user', 'orders.orderDetails.variation.product', 'orders.orderDetails.variation.combinations', 'orders.shop'])->first();
    if ($order) {
      if (auth('api')->user()->id == $order->user_id) {
        return new OrderSingleCollection($order);
      } else {
        return response()->json([
          'success' => false,
          'message' => translate("This order is not your. You can't check details of this order"),
          'status' => 200
        ]);
      }
    } else {
      return response()->json([
        'success' => false,
        'message' => translate("No order found by this code"),
        'status' => 404
      ]);
    }
  }

  public function get_shipping_cost(Request $request, $address_id)
  {
    $address = Address::find($address_id);
    $city = City::find($address->city_id);

    if ($city && $city->zone != null) {
      return response()->json([
        'success' => true,
        'standard_delivery_cost' => $city->zone->standard_delivery_cost,
        'express_delivery_cost' => $city->zone->express_delivery_cost,
      ]);
    } else {
      return response()->json([
        'success' => false,
        'standard_delivery_cost' => 0,
        'express_delivery_cost' => 0,
      ]);
    }
  }

  public function cancel($order_id)
  {
    $order = Order::findOrFail($order_id);
    if (auth('api')->user()->id !== $order->user_id) {
      return response()->json(null, 401);
    }

    if ($order->delivery_status == 'order_placed' && $order->payment_status == 'unpaid') {
      $order->delivery_status = 'cancelled';
      $order->save();

      foreach ($order->orderDetails as $orderDetail) {
        try {
          foreach ($orderDetail->product->categories as $category) {
            $category->sales_amount -= $orderDetail->total;
            $category->save();
          }

          $brand = $orderDetail->product->brand;
          if ($brand) {
            $brand->sales_amount -= $orderDetail->total;
            $brand->save();
          }
        } catch (\Exception $e) {
        }
      }

      return response()->json([
        'success' => true,
        'message' => translate("Order has been cancelled"),
      ]);
    } else {
      return response()->json([
        'success' => false,
        'message' => translate("This order can't be cancelled."),
      ]);
    }
  }

  public function store(Request $request)
  {

    $combined_order = new CombinedOrder;
    $combined_order->code = date('Ymd-His') . rand(10, 99);

    $stateCode = $request->billing['state'];
    $state = State::where('state_code', $stateCode)->first();

    $shippingAddress = $request->billing['address_1'] . ', ' . $request->billing['address_2'] . ', ' . $request->billing['city'] . ', ' . $state->name;
    $combined_order->shipping_address = $shippingAddress;
    $combined_order->billing_address = $combined_order->shipping_address;

    $grand_total = 0; // Initialize grand_total
    $package_number = 1;
    $shop_total = 0;

    $modified_line_items = [];

    foreach ($request->line_items as $line_item) {
      $product_id = $line_item['product_id'];
      $quantity = $line_item['quantity'];

      // Retrieve product details using $product_id
      $product = Product::find($product_id);

      // Calculate the price for this product
      $product_price = $product->lowest_price;

      // Calculate the total price for this line item
      $line_item_total = $product_price * $quantity;

      // Create a new modified line item
      $modified_line_item = $line_item;
      $modified_line_item['price'] = $product_price; // Update the price field in the line item
      $modified_line_item['total'] = $line_item_total; // Update the total field in the line item

      // Add the modified line item to the array
      $modified_line_items[] = $modified_line_item;

      // Add the line item's total to the grand total
      $grand_total += $line_item_total;
    }

    // Replace the original line_items property with the modified line items
    $request->merge(['line_items' => $modified_line_items]);

    // if ($request->delivery_type == 'standard') {
    //     $shipping_cost = $shippingCity->zone->standard_delivery_cost;
    // } elseif ($request->delivery_type == 'express') {
    //     $shipping_cost = $shippingCity->zone->express_delivery_cost;
    // }

    $delivery_type = $request->shipping_lines[0]['method_title']; //$delivery_type = 'express';
    $shipping_cost = floatval($request->shipping_total); //$shipping_cost = 20.00; //$shippingCity->zone->express_delivery_cost;

    $grand_total += $shipping_cost;

    $combined_order->grand_total = $grand_total;
    $combined_order->save();

    // Find the meta_data entry with key "_billing_seller_id"
    $billing_seller_meta = collect($request->meta_data)->firstWhere('key', '_billing_seller_id');

    // Extract the value from the meta_data entry
    $billing_seller_id = $billing_seller_meta['value'];

    // shop order place
    $order = Order::create([
      'shop_id' => $billing_seller_id,
      'combined_order_id' => $combined_order->id,
      'name' => $request->billing['first_name'] . ' ' . $request->billing['last_name'],
      'email' => $request->billing['email'],
      'phone' => $request->billing['phone'],
      'code' => $package_number,
      'shipping_address' => $combined_order->shipping_address,
      'billing_address' => $combined_order->shipping_address,
      'customer_city' => $request->billing['city'],
      'customer_state' => $state->name,
      'shipping_cost' => $shipping_cost,
      'grand_total' => $grand_total,
      'delivery_type' => $delivery_type,
      'payment_type' => $request->payment_method_title,
    ]);

    $package_number++;

    $order_price = $order->grand_total - $order->shipping_cost;

    \Log::info($billing_seller_id);

    $shop_commission = Shop::find($billing_seller_id)->commission;
    \Log::info($shop_commission);
    $seller_earning = $grand_total; // Use grand_total here instead of shop_total
    if ($shop_commission > 0) {
      $seller_earning = ($shop_commission * $order_price) / 100;
    }


    $order->seller_earning = $seller_earning;
    $order->commission_percentage = $shop_commission;
    $order->save();
    OrderUpdate::create([
      'order_id' => $order->id,
      'user_id' => '1',
      'note' => 'Order has been placed.',
    ]);

    // Create OrderDetail records for each product within the order
    foreach ($request->line_items as $line_item) {
      $product_id = $line_item['product_id'];
      $quantity = $line_item['quantity'];

      // Retrieve product details using $product_id
      $product = Product::find($product_id);

      // Calculate the price for this product and add it to the total
      $product_price = $product->lowest_price;
      $shop_total += $product_price * $quantity;

      // Create an OrderDetail record for this product
      $orderDetail = OrderDetail::create([
        'order_id' => $order->id,
        'product_id' => $product_id,
        'price' => $product_price,
        'quantity' => $quantity,
        'total' => $product_price * $quantity,
      ]);
    }

    try {
      //Notification::send($user, new OrderPlacedNotification($combined_order));
      $userEmail = $request->email;
      Notification::route('mail', $userEmail)->notify(new OrderPlacedNotification($combined_order));
      foreach ($combined_order->orders as $order) {
        $seller = $order->orderDetails->first()->product->shop;
        Notification::send($seller, new SellerInvoiceNotification($order));
        Notification::send($seller, new OrderDbNotification($order));
      }
    } catch (\Exception $e) {
    }

    return response()->json([
      'success' => true,
    ]);
  }

  public function paymentDone($combined_order, $payment_method, $payment_info = null)
  {
    foreach ($combined_order->orders as $order) {

      // commission calculation
      calculate_seller_commision($order);

      $order->payment_status = 'paid';
      $order->payment_type = $payment_method;
      $order->payment_details = $payment_info;
      $order->save();
    }
  }

  public function productDownloads()
  {
    $products = DB::table('orders')
      ->orderBy('id', 'desc')
      ->join('order_details', 'orders.id', '=', 'order_details.order_id')
      ->join('products', 'order_details.product_id', '=', 'products.id')
      ->where('orders.user_id', auth()->id())
      ->where('products.digital', '1')
      ->where('orders.payment_status', 'paid')
      ->select('products.*')
      ->paginate(15);

    return DownloadedProductResource::collection($products)->additional([
      'success' => true,
      'status' => 200
    ]);
  }
  public function download($product)
  {
    $product = Product::findOrFail(($product));

    $upload = Upload::findOrFail($product->file_name);

    if (env('FILESYSTEM_DRIVER') == "s3") {
      return \Storage::disk('s3')->download($upload->file_name, $upload->file_original_name . "." . $upload->extension);
    } else {

      if (file_exists(base_path('public/' . $upload->file_name))) {
        return response()->download($_SERVER['HTTP_HOST'] . 'public/' . $upload->file_name);
      }
    }
  }
}
