<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Category;
use App\Models\CategoryTranslation;
use App\Utility\CategoryUtility;
use App\Models\CommissionHistory;
use App\Models\SellerPackage;          // ADD
use Carbon\Carbon;                     // ADD

class CategoryController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_categories'])->only('index');
    $this->middleware(['permission:add_categories'])->only('create', 'store', 'storeAjax');
    $this->middleware(['permission:edit_categories'])->only('edit', 'update', 'updateFeatured');
    $this->middleware(['permission:delete_categories'])->only('destroy');
  }

  public function index(Request $request)
  {
    $sort_search = null;
    $q = Category::orderBy('created_at', 'desc');

    if ($request->filled('search')) {
      $sort_search = $request->string('search');
      $q->where('name', 'like', '%' . $sort_search . '%');
    }

    $categories = $q->paginate(12);

    return view('backend.product.categories.index', compact('categories', 'sort_search'));
  }

  public function create()
  {
    $categories = Category::where('parent_id', 0)
      ->with('childrenCategories')
      ->get();

    return view('backend.product.categories.create', compact('categories'));
  }

  public function store(Request $request)
  {
    $validated = $request->validate([
      'name'                   => 'required|string|max:191',
      'parent_id'              => 'nullable|integer|exists:categories,id',
      'order_level'            => 'nullable|integer|min:0',
      'banner'                 => 'nullable',
      'icon'                   => 'nullable',
      'meta_title'             => 'nullable|string|max:255',
      'meta_image'             => 'nullable',
      'meta_description'       => 'nullable|string',
      'slug'                   => 'nullable|string|max:191',
      'digital'                => 'nullable|in:0,1',
      'filtering_attributes'   => 'nullable|array',
      'filtering_attributes.*' => 'integer|exists:attributes,id',
    ]);

    $category = new Category();
    $category->name              = $validated['name'];
    $category->order_level       = (int) ($validated['order_level'] ?? 0);
    $category->banner            = $validated['banner'] ?? null;
    $category->icon              = $validated['icon'] ?? null;
    $category->meta_title        = $validated['meta_title'] ?? null;
    $category->meta_image        = $validated['meta_image'] ?? null;
    $category->meta_description  = $validated['meta_description'] ?? null;

    if (!empty($validated['parent_id'])) {
      $category->parent_id = (int) $validated['parent_id'];
      $parent = Category::find($category->parent_id);
      $category->level = ($parent?->level ?? 0) + 1;
    } else {
      $category->parent_id = 0;
      $category->level = 0;
    }

    if ($request->filled('slug')) {
      $category->slug = Str::slug($request->string('slug'), '-');
    }

    $category->digital = (int) ($validated['digital'] ?? 0);

    $category->save();

    $category->attributes()->sync($validated['filtering_attributes'] ?? []);

    $catTrans = CategoryTranslation::firstOrNew([
      'lang'        => env('DEFAULT_LANGUAGE'),
      'category_id' => $category->id,
    ]);
    $catTrans->name = $validated['name'];
    $catTrans->save();

    flash(translate('Category has been inserted successfully'))->success();
    return redirect()->route('categories.index');
  }

  public function edit(Request $request, $id)
  {
    $lang = $request->string('lang', env('DEFAULT_LANGUAGE'));
    $category = Category::findOrFail($id);

    $categories = Category::where('parent_id', 0)
      ->with('childrenCategories')
      ->whereNotIn('id', CategoryUtility::children_ids($category->id, true))
      ->where('id', '!=', $category->id)
      ->orderBy('name', 'asc')
      ->get();

    return view('backend.product.categories.edit', compact('category', 'categories', 'lang'));
  }

  public function update(Request $request, $id)
  {
    $category = Category::findOrFail($id);

    $validated = $request->validate([
      'name'                   => 'required|string|max:191',
      'parent_id'              => 'nullable|integer|exists:categories,id',
      'order_level'            => 'nullable|integer|min:0',
      'banner'                 => 'nullable',
      'icon'                   => 'nullable',
      'meta_title'             => 'nullable|string|max:255',
      'meta_image'             => 'nullable',
      'meta_description'       => 'nullable|string',
      'slug'                   => 'nullable|string|max:191',
      'digital'                => 'nullable|in:0,1',
      'lang'                   => 'nullable|string',
      'filtering_attributes'   => 'nullable|array',
      'filtering_attributes.*' => 'integer|exists:attributes,id',
    ]);

    if (($validated['lang'] ?? env('DEFAULT_LANGUAGE')) === env('DEFAULT_LANGUAGE')) {
      $category->name = $validated['name'];
    }

    $category->order_level       = (int) ($validated['order_level'] ?? $category->order_level ?? 0);
    $category->banner            = $validated['banner'] ?? null;
    $category->icon              = $validated['icon'] ?? null;
    $category->meta_title        = $validated['meta_title'] ?? null;
    $category->meta_image        = $validated['meta_image'] ?? null;
    $category->meta_description  = $validated['meta_description'] ?? null;

    $previous_level = $category->level ?? 0;

    if (!empty($validated['parent_id'])) {
      $category->parent_id = (int) $validated['parent_id'];
      $parent = Category::find($category->parent_id);
      $category->level = ($parent?->level ?? 0) + 1;
    } else {
      $category->parent_id = 0;
      $category->level = 0;
    }

    if ($category->level > $previous_level) {
      CategoryUtility::move_level_down($category->id);
    } elseif ($category->level < $previous_level) {
      CategoryUtility::move_level_up($category->id);
    }

    $category->digital = (int) ($validated['digital'] ?? 0);

    if ($request->filled('slug')) {
      $category->slug = Str::slug($request->string('slug'), '-');
    }

    $category->save();

    $category->attributes()->sync($validated['filtering_attributes'] ?? []);

    $lang = $validated['lang'] ?? env('DEFAULT_LANGUAGE');
    $catTrans = CategoryTranslation::firstOrNew([
      'lang'        => $lang,
      'category_id' => $category->id,
    ]);
    $catTrans->name = $validated['name'];
    $catTrans->save();

    flash(translate('Category has been updated successfully'))->success();
    return back();
  }

  public function destroy($id)
  {
    $category = Category::findOrFail($id);

    $category->product_categories()->delete();
    CategoryUtility::delete_category($id);

    flash(translate('Category has been deleted successfully'))->success();

    $returnTo = request('return_to');
    if ($returnTo && filter_var($returnTo, FILTER_VALIDATE_URL)) {
      return redirect()->to($returnTo);
    }
    return redirect()->back();
  }

  public function updateFeatured(Request $request)
  {
    $request->validate([
      'id'     => 'required|integer|exists:categories,id',
      'status' => 'required|in:0,1',
    ]);

    $category = Category::findOrFail($request->id);
    $category->featured = (int) $request->status;
    return $category->save() ? 1 : 0;
  }

  /**
   * AJAX endpoint used by your product create blade.
   * Returns: { id, name, slug }
   */
  public function storeAjax(Request $request)
  {
    $validated = $request->validate([
      'name'      => 'required|string|max:191',
      'parent_id' => 'nullable|integer|exists:categories,id',
    ]);

    $category = Category::create([
      'name'      => $validated['name'],
      'parent_id' => $validated['parent_id'] ?? 0,
    ]);

    return response()->json([
      'id'   => $category->id,
      'name' => $category->name,
      'slug' => $category->slug,
    ]);
  }

  public function product_category_dashboard($slug)
  {
    if (!auth()->check() || !in_array(auth()->user()->user_type, ['merchant', 'seller'], true)) {
      abort(403);
    }

    $user     = auth()->user();
    $category = \App\Models\Category::where('slug', $slug)->firstOrFail();

    // Include descendants
    $descendantIds = \App\Utility\CategoryUtility::children_ids($category->id, true);
    $categoryIds   = array_values(array_unique(array_merge([$category->id], $descendantIds ?: [])));

    // All product IDs that belong to the current category or any descendant
    $productIds = \DB::table('product_categories')
      ->whereIn('category_id', $categoryIds)
      ->pluck('product_id')
      ->unique()
      ->values();

    // Paginated products list for the grid
    $products = \App\Models\Product::where('published', 1)
      ->whereIn('id', $productIds->isEmpty() ? [-1] : $productIds)
      ->with('categories', 'tags')
      ->paginate(20);

    // ---- Effective commissions for each product (override in window → else package default) ----
    $pkg = $user->package_id ? SellerPackage::find($user->package_id) : null;
    $now = Carbon::now();

    $products->getCollection()->transform(function ($p) use ($pkg, $now) {
      $overrideActive = ((int)($p->is_commission_override_active ?? 0) === 1)
        && (!$p->commission_effective_from || $now->gte(Carbon::parse($p->commission_effective_from)))
        && (!$p->commission_effective_to   || $now->lte(Carbon::parse($p->commission_effective_to)));

      $pick = static function ($oType, $oVal, $dType, $dVal, $overrideActive) {
        if ($overrideActive && $oType && $oVal !== null) {
          return ['type' => $oType, 'value' => (float)$oVal, 'source' => 'override'];
        }
        if ($dType && $dVal !== null) {
          return ['type' => $dType, 'value' => (float)$dVal, 'source' => 'default'];
        }
        return ['type' => null, 'value' => null, 'source' => null];
      };

      // Reseller → package.commission[_type]
      $p->eff_reseller = $pick(
        $p->reseller_type,
        $p->reseller_value,
        optional($pkg)->commission_type,
        optional($pkg)->commission,
        $overrideActive
      );

      // Referral Sale → package.ref_sale_commission[_type]
      $p->eff_referral = $pick(
        $p->referral_type,
        $p->referral_value,
        optional($pkg)->ref_sale_commission_type,
        optional($pkg)->ref_sale_commission,
        $overrideActive
      );

      // Area Bonus → package.area_sale_bonus[_type]
      $p->eff_area = $pick(
        $p->area_type,
        $p->area_value,
        optional($pkg)->area_sale_bonus_type,
        optional($pkg)->area_sale_bonus,
        $overrideActive
      );

      $p->override_active = $overrideActive;
      return $p;
    });
    // ----------------------------------------------------------------------

    // Orders that contain those products
    $orderIds = \DB::table('order_details')
      ->whereIn('product_id', $productIds->isEmpty() ? [-1] : $productIds)
      ->pluck('order_id')
      ->unique()
      ->values();

    // Default widgets
    $saleCommision  = 0.0;
    $todayEarnings  = 0.0;
    $allTime        = 0.0;
    $currentBalance = 0.0;

    if ($orderIds->isNotEmpty()) {
      // Per-order totals inside this category
      $catTotalsByOrder = \DB::table('order_details')
        ->select('order_id', \DB::raw('SUM(total) as cat_total'))
        ->whereIn('product_id', $productIds)
        ->groupBy('order_id')
        ->pluck('cat_total', 'order_id');

      // Per-order totals across entire order
      $ordTotalsByOrder = \DB::table('order_details')
        ->select('order_id', \DB::raw('SUM(total) as ord_total'))
        ->whereIn('order_id', $orderIds)
        ->groupBy('order_id')
        ->pluck('ord_total', 'order_id');

      // Share helper
      $getShare = function ($orderId) use ($catTotalsByOrder, $ordTotalsByOrder) {
        $cat = (float) ($catTotalsByOrder->get($orderId, 0) ?? 0);
        $all = (float) ($ordTotalsByOrder->get($orderId, 0) ?? 0);
        return $all > 0 ? max(0.0, min(1.0, $cat / $all)) : 0.0;
      };

      // Pull commission rows once
      $rows = \App\Models\CommissionHistory::where('user_id', $user->id)
        ->whereIn('order_id', $orderIds)
        ->get();

      // Category gross by proportional share per order
      $grossCategory = $rows->sum(function ($r) use ($getShare) {
        return (float) $r->seller_earning * $getShare($r->order_id);
      });

      // Sale commission (order-only) with share
      $saleCommision = $rows
        ->filter(function ($r) {
          return $r->source_type === 'order_sale' || $r->details === 'Order Payment.';
        })
        ->sum(function ($r) use ($getShare) {
          return (float) $r->seller_earning * $getShare($r->order_id);
        });

      // Today’s earnings with share
      $todayEarnings = $rows
        ->filter(function ($r) {
          return $r->created_at && $r->created_at->isToday();
        })
        ->sum(function ($r) use ($getShare) {
          return (float) $r->seller_earning * $getShare($r->order_id);
        });

      // Alias for blade
      $allTime = $grossCategory;

      // Proportional category “Current Balance”
      $grossAllTime = (float) \App\Models\CommissionHistory::where('user_id', $user->id)->sum('seller_earning');

      $paidPayouts = (float) \DB::table('seller_payouts')
        ->where('user_id', $user->id)
        ->where('status', 'paid')
        ->sum(\DB::raw('COALESCE(paid_amount, requested_amount, 0)'));

      $globalShare    = $grossAllTime > 0 ? ($grossCategory / $grossAllTime) : 0.0;
      $allocatedPaid  = round($paidPayouts * $globalShare, 2);
      $currentBalance = max(0, round($grossCategory - $allocatedPaid, 2));
    }

    // Global buckets (optional; remove if not displayed)
    $referralCommision = (float) \App\Models\CommissionHistory::where('user_id', $user->id)
      ->where(function ($q) {
        $q->where('source_type', 'registration')
          ->orWhere('details', 'Package Payment.');
      })
      ->sum('seller_earning');

    $referralSaleCommision = (float) \App\Models\CommissionHistory::where('user_id', $user->id)
      ->where(function ($q) {
        $q->where('source_type', 'referral_sale')
          ->orWhere('details', 'Referral Sale Commission.');
      })
      ->sum('seller_earning');

    $areaSaleBonus = (float) \App\Models\CommissionHistory::where('user_id', $user->id)
      ->where(function ($q) {
        $q->where('source_type', 'area_bonus')
          ->orWhere('details', 'Area Sale Bonus.');
      })
      ->sum('seller_earning');

    return view('backend.product.products.category_products', compact(
      'category',
      'products',
      'saleCommision',         // category-scoped by share
      'referralCommision',     // global (optional)
      'referralSaleCommision', // global (optional)
      'areaSaleBonus',         // global (optional)
      'todayEarnings',         // category-scoped by share
      'currentBalance',        // category-scoped proportional net
      'allTime'                // category-scoped gross by share
    ));
  }
}
