<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Models\Country;
use App\Models\City;
use App\Models\State;
use App\Models\Region;
use App\Models\Barangay;
use App\Models\SellerPackage;
use App\Models\SellerPackagePayment;
use App\Notifications\SellerRegistrationNotification;
use App\Notifications\SellerReferralRegistrationNotification;

class HomeController extends Controller
{
  public function index(Request $request)
  {
    return view('frontend.home');
  }

  public function signup(Request $request)
  {
    $selleridValidator = Validator::make(
      $request->only('sellerid'),
      ['sellerid' => 'required|exists:users,id'],
      [
        'sellerid.required' => 'This link is invalid. Please check your link and try again.',
        'sellerid.exists'   => 'This link is invalid. Please check your link and try again.',
      ]
    );
    if ($selleridValidator->fails()) {
      $msg = $selleridValidator->errors()->first('sellerid');
      flash(translate($msg))->error();
      return back()->withErrors($selleridValidator)->withInput();
    }

    $baseValidator = Validator::make(
      $request->all(),
      [
        'first_name' => 'required|string|max:255',
        'last_name'  => 'required|string|max:255',
        'email'      => [
          'required',
          'string',
          'email',
          'max:255',
          'unique:users',
          function ($attr, $val, $fail) {
            if (!preg_match('/^[a-zA-Z0-9._%+-]+@(gmail|yahoo|microsoft)\.(com|net|org)$/i', $val)) {
              $fail('Only Gmail, Yahoo, or Microsoft emails are allowed.');
            }
          },
        ],
        'phone'      => 'required|string|regex:/^[0-9]{11}$/',
        'password'   => 'required|string|min:6|max:20|confirmed',
        'membership' => 'required|exists:seller_packages,id',
      ],
      [
        'email.unique'       => 'The email address has already been registered. Please use a different email.',
        'email.regex'        => 'Only Gmail, Yahoo, or Microsoft emails are allowed.',
        'password.min'       => 'The password must be at least 6 characters.',
        'password.max'       => 'The password may not be greater than 20 characters.',
        'password.confirmed' => 'The password confirmation does not match.',
      ]
    );
    if ($baseValidator->fails()) {
      return back()->withErrors($baseValidator)->withInput();
    }

    $package = SellerPackage::findOrFail($request->membership);
    $level   = (int) ($package->geographic_level ?? 0);

    $locationRules = [];
    switch ($level) {
      case 1:
        $locationRules['selected_city'] = 'required|exists:cities,id';
        $locationRules['selected_barangay'] = 'required|exists:barangays,id';
        break;
      case 2:
        $locationRules['selected_state'] = 'required|exists:states,id';
        $locationRules['selected_city'] = 'required|exists:cities,id';
        break;
      case 3:
        $locationRules['selected_state'] = 'required|exists:states,id';
        break;
      case 4:
        $locationRules['selected_region'] = 'required|exists:regions,id';
        break;
      case 5:
        $locationRules['selected_country'] = 'required|exists:countries,id';
        break;
    }
    if (!empty($locationRules)) {
      $locValidator = Validator::make($request->all(), $locationRules);
      if ($locValidator->fails()) return back()->withErrors($locValidator)->withInput();
    }

    DB::beginTransaction();
    try {
      $user = new User();
      $user->name              = trim($request->first_name . ' ' . $request->last_name);
      $user->email             = $request->email;
      $user->phone             = $request->phone;
      $user->user_type         = 'seller';
      $user->referred_by       = $request->sellerid;
      $user->password          = Hash::make($request->password);
      $user->email_verified_at = now()->format('Y-m-d H:i:s');
      $user->save();

      $set_payment = null;
      $needsPayment = false;

      if ($level === 0) {
        $user->approval = 0;
        $user->save();
      } else {
        $user->approval = 0;
        $user->save();

        $payment = new SellerPackagePayment();
        $payment->user_id           = $user->id;
        $payment->seller_package_id = $request->membership;
        $payment->amount            = (float) ($package->amount ?? 0);
        $payment->approval          = 0;
        $payment->save();

        $payment->transaction_id = $payment->id
          . '-' . $user->id
          . '-' . $request->membership
          . '-' . \Illuminate\Support\Str::upper(\Illuminate\Support\Str::random(6));
        $payment->save();

        $set_payment = $payment;
        $needsPayment = true;
      }

      $user->verification_status = 0;
      $user->package_id          = $package->id;
      $user->seller_type         = strtolower($package->name ?? '');
      $user->save();

      $reqRegion   = null;
      $reqState    = null;
      $reqCity     = null;
      $reqBarangay = null;
      $reqCountry  = null;

      switch ($level) {
        case 1:
          $reqCity = (int)$request->selected_city;
          $reqBarangay = (int)$request->selected_barangay;
          break;
        case 2:
          $reqState = (int)$request->selected_state;
          $reqCity = (int)$request->selected_city;
          break;
        case 3:
          $reqState = (int)$request->selected_state;
          break;
        case 4:
          $reqRegion = (int)$request->selected_region;
          break;
        case 5:
          $reqCountry = (int)$request->selected_country;
          break;
      }

      if ($set_payment) {
        $set_payment->requested_region_id    = $reqRegion;
        $set_payment->requested_state_id     = $reqState;
        $set_payment->requested_city_id      = $reqCity;
        $set_payment->requested_barangay_id  = $reqBarangay;
        $set_payment->requested_country_id   = $reqCountry;

        $locId = null;
        switch ($level) {
          case 1:
            $locId = $reqBarangay ?: $reqCity;
            break;
          case 2:
            $locId = $reqCity;
            break;
          case 3:
            $locId = $reqState;
            break;
          case 4:
            $locId = $reqRegion;
            break;
          case 5:
            $locId = $reqCountry;
            break;
        }
        $set_payment->seller_assign_location = $locId;
        $set_payment->save();
      }

      $user->seller_assign_location = null;
      $user->save();

      try {
        $admin = User::where('user_type', 'admin')->first();
        if ($admin) $admin->notify(new SellerRegistrationNotification($user));
      } catch (\Throwable $e) {
      }
      try {
        $user->notify(new SellerReferralRegistrationNotification($user));
      } catch (\Throwable $e) {
      }

      DB::commit();

      return redirect()
        ->route('thank-you-for-registration')
        ->with('pending_approval_only', $level === 0)
        ->with('needs_payment', $needsPayment);
    } catch (\Throwable $e) {
      DB::rollBack();
      flash(translate('Something went wrong while creating your account. Please try again.'))->error();
      return back()->withInput();
    }
  }

  // Lookups (single-owner only)
  public function getStates(Request $request)
  {
    $level = SellerPackage::find($request->query('mempckge'))?->geographic_level ?? 0;
    if ($level < 2) return response()->json(['states' => []]);

    $states = State::query()
      ->where('status', 1)
      ->whereNull('seller_assign')
      ->orderBy('name')
      ->get(['id', 'name']);

    return response()->json(['states' => $states]);
  }

  public function getCities(Request $request, $stateId = null)
  {
    $level = SellerPackage::find($request->query('mempckge'))?->geographic_level ?? 0;
    if ($level < 1 || !$stateId) return response()->json(['cities' => []]);

    $q = City::query()->where('status', 1)->where('state_id', $stateId);
    if ($level === 2) $q->whereNull('seller_assign');

    $cities = $q->orderBy('name')->get(['id', 'name']);
    return response()->json(['cities' => $cities]);
  }

  public function getBarangays(Request $request, $cityId)
  {
    $level = SellerPackage::find($request->query('mempckge'))?->geographic_level ?? 0;
    if ($level !== 1) return response()->json(['barangays' => []]);

    $barangays = Barangay::query()
      ->where('city_id', $cityId)
      ->where('status', 1)
      ->whereNull('seller_assign')
      ->orderBy('name')
      ->get(['id', 'name']);

    return response()->json(['barangays' => $barangays]);
  }

  public function getRegions(Request $request)
  {
    $level = SellerPackage::find($request->query('mempckge'))?->geographic_level ?? 0;
    if ($level != 4) return response()->json(['regions' => []]);

    $regions = Region::query()
      ->where('status', 1)
      ->whereNull('seller_assign')
      ->orderBy('name')
      ->get(['id', 'name']);

    return response()->json(['regions' => $regions]);
  }

  public function getCountries(Request $request)
  {
    $level = SellerPackage::find($request->query('mempckge'))?->geographic_level ?? 0;
    if ($level !== 5) return response()->json(['countries' => []]);

    $countries = Country::query()
      ->where('status', 1)
      ->orderBy('name')
      ->get(['id', 'name']);

    return response()->json(['countries' => $countries]);
  }

  public function membershipregistration(Request $request)
  {
    $mempckge = $request->query('mempckge');

    $packages = SellerPackage::all();
    $selectedPackage = $mempckge ? SellerPackage::find($mempckge) : null;
    $geographicLevel = $selectedPackage?->geographic_level ?? 0;

    $states = $cities = $barangays = $regions = collect();

    if ($geographicLevel > 0) $states = State::where('status', 1)->get();
    if ($request->has('selected_state') && $geographicLevel >= 1) {
      $cities = City::where('status', 1)->where('state_id', $request->input('selected_state'))->get();
    }
    if ($request->has('selected_city') && $geographicLevel >= 1) {
      $barangays = Barangay::where('city_id', $request->input('selected_city'))->get();
    }
    if ($geographicLevel == 4) $regions = Region::where('status', 1)->get();

    return view('frontend.membership', compact('packages', 'states', 'cities', 'barangays', 'regions', 'geographicLevel'));
  }

  public function thankyouforregistration(Request $request)
  {
    return view('frontend.thankyouforregistration');
  }
}
