<?php

namespace App\Http\Controllers\Merchant;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class EarningController extends Controller
{
    public function index(Request $request)
    {
        $uid = auth()->id();

        // Scope products owned by this merchant via merchant_id OR user_id
        $myProductIds = Product::query()
            ->where(function ($q) use ($uid) {
                $q->where('merchant_id', $uid)
                  ->orWhere('user_id', $uid);
            })
            ->pluck('id');

        // Guard empty set to avoid returning all rows or none incorrectly
        $pidSet = $myProductIds->isNotEmpty() ? $myProductIds : collect([-1]);

        // Orders that contain any of the merchant's products
        $orderIdsForMe = OrderDetail::query()
            ->whereIn('product_id', $pidSet)
            ->pluck('order_id')
            ->unique()
            ->values();

        $oidSet = $orderIdsForMe->isNotEmpty() ? $orderIdsForMe : collect([-1]);

        // Totals
        $ordersCount = Order::whereIn('id', $oidSet)->count();

        // All-time subtotal (price*qty) for this merchant (regardless of payment)
        $ordersAmountAll = (float) OrderDetail::query()
            ->whereIn('product_id', $pidSet)
            ->sum(DB::raw('price * quantity'));

        // Paid orders only — match the job’s paid policy
        $completedIds = Order::whereIn('id', $oidSet)
            ->where(function ($q) {
                $q->whereRaw('LOWER(payment_status) = ?', ['paid'])
                  ->orWhereIn('payment_status', [1, '1', true]);
            })
            ->pluck('id');

        $cidSet = $completedIds->isNotEmpty() ? $completedIds : collect([-1]);
        $completedCount = $completedIds->count();

        // Gross paid, subtotal only
        $grossPaid = (float) OrderDetail::query()
            ->whereIn('product_id', $pidSet)
            ->whereIn('order_id', $cidSet)
            ->sum(DB::raw('price * quantity'));

        $commissionRate   = (float) (get_setting('merchant_commission_rate') ?? 10);
        $commissionFactor = $commissionRate / 100;
        $adminEarning     = $completedCount > 0 ? round($grossPaid * $commissionFactor, 2) : 0.0;
        $net              = $completedCount > 0 ? max(0, $grossPaid - $adminEarning) : 0.0;

        // Per-product (PAID only), subtotal only — stays as a query (not a collection)
        $perProduct = OrderDetail::with('product:id,name,user_id,merchant_id')
            ->whereIn('product_id', $pidSet)
            ->whereIn('order_id', $cidSet)
            ->selectRaw('product_id, SUM(price * quantity) as gross, SUM(quantity) as qty')
            ->groupBy('product_id')
            ->orderByDesc(DB::raw('SUM(price * quantity)'))
            ->paginate(15);

        // Orders table (optional date range) — includes ALL orders touching this merchant
        $date_range = $request->get('date_range');

        $ordersQuery = Order::with(['combined_order', 'user'])
            ->whereIn('id', $oidSet);

        $hasTo = function_exists('str_contains')
            ? (fn($s) => str_contains($s, ' to '))
            : (fn($s) => strpos($s, ' to ') !== false);

        if (!empty($date_range) && $hasTo($date_range)) {
            [$from, $to] = array_map('trim', explode(' to ', $date_range, 2));
            if ($from && $to) {
                $ordersQuery->whereDate('created_at', '>=', $from)
                            ->whereDate('created_at', '<=', $to);
            }
        }

        $orders = $ordersQuery->latest('id')
            ->paginate(15)
            ->appends($request->query());

        // Per-order stats (subtotal & item count) for THIS merchant
        $stats = $orders->isEmpty()
            ? collect()
            : OrderDetail::query()
                ->whereIn('order_id', $orders->pluck('id'))
                ->whereIn('product_id', $pidSet)
                ->selectRaw('order_id, COUNT(*) as items_count, SUM(price * quantity) as amount')
                ->groupBy('order_id')
                ->get()
                ->keyBy('order_id');

        return view('backend.merchants.earnings.index', [
            'ordersCount'     => $ordersCount,
            'ordersAmountAll' => $ordersAmountAll,
            'completedCount'  => $completedCount,
            'gross'           => $grossPaid,
            'adminEarning'    => $adminEarning,
            'net'             => $net,
            'perProduct'      => $perProduct,
            'orders'          => $orders,
            'orderStats'      => $stats,
            'date_range'      => $date_range,
            'commissionRate'  => $commissionRate,
        ]);
    }
}
