<?php

namespace App\Http\Controllers\Merchant;

use App\Http\Controllers\Controller;
use App\Models\SellerPayout;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class PayoutController extends Controller
{
  public function __construct()
  {
    $this->middleware(['auth', 'permission:merchant.view_payouts'])->only('index', 'requestIndex');
    $this->middleware(['auth', 'permission:merchant.request_payout'])->only('storeRequest');
    $this->middleware(['auth', 'permission:merchant.payout_settings'])->only('settings', 'settingsUpdate');
  }

  public function index(Request $request)
  {
    $date_range = $request->date_range;
    $q = SellerPayout::where('user_id', Auth::id())->where('status', 'paid');
    if ($date_range) {
      [$from, $to] = array_map('trim', explode(' / ', $date_range));
      $q->whereBetween('created_at', [Carbon::parse($from)->startOfDay(), Carbon::parse($to)->endOfDay()]);
    }
    $payouts = $q->latest()->paginate(20);
    return view('backend.merchants.payouts.index', compact('payouts', 'date_range'));
  }

  public function requestIndex()
  {
    $user = auth()->user();
    $payout_requests = SellerPayout::where('user_id', $user->id)->where('status', 'requested')->latest()->paginate(15);
    $currentBalance = (float) ($user->current_balance ?? 0);
    $gcashEnabled = (int) ($user->gcash_payout_status ?? 0) === 1 && filled($user->gcash_name) && filled($user->gcash_no);
    $bankEnabled  = (int) ($user->bank_payout_status ?? 0)  === 1 && filled($user->bank_name) && filled($user->bank_acc_name) && filled($user->bank_acc_no);
    $canRequest = $gcashEnabled || $bankEnabled;
    return view('backend.merchants.payouts.requests', compact('payout_requests', 'currentBalance', 'gcashEnabled', 'bankEnabled', 'canRequest', 'user'));
  }

  public function storeRequest(Request $request)
  {
    $request->validate([
      'amount'         => ['required', 'numeric', 'min:1'],
      'message'        => ['nullable', 'string', 'max:2000'],
      'payment_method' => ['required', 'in:gcash,bank'],
    ]);
    $user = auth()->user();

    if ($request->payment_method === 'gcash') {
      abort_unless((int)($user->gcash_payout_status ?? 0) === 1 && filled($user->gcash_name) && filled($user->gcash_no), 422, 'GCash not configured.');
    } else {
      abort_unless((int)($user->bank_payout_status ?? 0) === 1 && filled($user->bank_name) && filled($user->bank_acc_name) && filled($user->bank_acc_no), 422, 'Bank not configured.');
    }

    $min = (float) get_setting('min_withdrawal_amount', 0);
    if ($min > 0 && (float)$request->amount < $min) {
      return back()->withErrors(['amount' => translate('Minimum withdrawal amount is :amount', ['amount' => format_price($min)])]);
    }

    $available = (float) ($user->current_balance ?? 0);
    if ($available < (float) $request->amount) {
      return back()->withErrors(['amount' => translate('You can not request more than your balance')]);
    }

    $details = $request->payment_method === 'gcash'
      ? ['gcash_name' => $user->gcash_name, 'gcash_no' => $user->gcash_no]
      : ['bank_name' => $user->bank_name, 'bank_acc_name' => $user->bank_acc_name, 'bank_acc_no' => $user->bank_acc_no];

    $p = new SellerPayout();
    $p->user_id          = $user->id;
    $p->requested_amount = (float) $request->amount;
    $p->seller_note      = $request->message;
    $p->status           = 'requested';
    $p->payment_method   = $request->payment_method;
    $p->payment_details  = json_encode($details);
    $p->save();

    return redirect()->route('merchant.payouts.request')->with('status', translate('Request has been sent successfully'));
  }

  public function settings()
  {
    $user = auth()->user();
    return view('backend.merchants.payouts.settings', compact('user'));
  }

  public function settingsUpdate(Request $request)
  {
    $user = auth()->user();
    $user->gcash_payout_status = $request->has('gcash_payout_status') ? 1 : 0;
    $user->bank_payout_status  = $request->has('bank_payout_status')  ? 1 : 0;
    $user->gcash_name = $request->gcash_name;
    $user->gcash_no   = $request->gcash_no;
    $user->bank_name     = $request->bank_name;
    $user->bank_acc_name = $request->bank_acc_name;
    $user->bank_acc_no   = $request->bank_acc_no;
    $user->save();
    
    flash(translate('Withdraw settings updated successfully'))->success();
    return back();
  }
}
