<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use App\Services\WooCommerceService;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductTranslation;
use App\Models\ProductVariation;
use App\Models\ProductVariationCombination;
use App\Models\ProductAttribute;
use App\Models\ProductAttributeValue;
use App\Models\Attribute;
use App\Models\AttributeValue;
use App\Models\Category;
use Illuminate\Validation\Rule;
use App\Models\ProductTax;
use App\Models\ShopBrand;
use App\Utility\CategoryUtility;
use CoreComponentRepository;
use Artisan;
use Carbon\Carbon;

class ProductController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_products'])->only('index');
    $this->middleware(['permission:add_products'])->only('create');
    $this->middleware(['permission:view_products'])->only('show');
    $this->middleware(['permission:edit_products'])->only('edit', 'update', 'updatePublished');
    $this->middleware(['permission:duplicate_products'])->only('duplicate');
    $this->middleware(['permission:delete_products'])->only('destroy');
  }

  // App\Http\Controllers\ProductController@index
  public function index(Request $request)
  {
    $col_name    = null;
    $query       = null;
    $sort_search = null;

    $products = Product::orderBy('created_at', 'desc');

    if ($request->filled('search')) {
      $sort_search = trim($request->search);
      $products->where('name', 'like', "%{$sort_search}%");
    }

    if ($request->filled('type')) {
      [$col_name, $query] = explode(',', $request->type);
      $products->orderBy($col_name, $query);
    }

    $products = $products->paginate(15);
    $type     = 'All';

    return view('backend.product.products.index', compact(
      'products',
      'type',
      'col_name',
      'query',
      'sort_search'
    ));
  }

  public function create()
  {
    $categories = Category::with('childrenCategories')
      ->where('parent_id', 0)
      ->get();

    return view('backend.product.products.create', compact('categories'));
  }

  public function store(Request $request, WooCommerceService $wc)
  {
    $request->validate([
      'name'  => ['required', 'string', 'max:200'],
      'price' => ['required', 'numeric', 'min:0'],
      'stock' => ['required', 'in:0,1'],
    ]);

    $product = new Product;
    $product->user_id       = auth()->id();
    $product->name          = $request->name;
    $product->unit          = $request->unit;
    $product->min_qty       = 1;
    $product->max_qty       = 0;
    $product->thumbnail_img = $request->thumbnail_img;
    $product->description   = $request->description;
    $product->published     = (string) $request->status === '1' ? 1 : 0;
    $product->stock         = $request->stock;
    $product->slug          = Str::slug($request->name, '-');
    $product->lowest_price  = $request->price;
    $product->highest_price = $request->price;

    $product->save();

    // categories
    if ($request->filled('category_ids')) {
      $ids = collect($request->category_ids)
        ->map(fn($id) => Category::find($id)?->id)
        ->filter()
        ->values()
        ->all();
      $product->categories()->sync($ids);
    }

    // Woo sync
    try {
      $wcCategories = [];
      foreach ($product->categories as $category) {
        $wcCat = $wc->createOrGetCategory($category->name);
        if (is_array($wcCat) && isset($wcCat['id'])) {
          $wcCategories[] = ['id' => $wcCat['id']];
        }
      }

      $payload = [
        'name'          => $product->name,
        'type'          => 'simple',
        'status'        => ((int) $product->published === 1 ? 'publish' : 'draft'),
        'regular_price' => (string) $product->lowest_price,
        'description'   => $product->description,
        'manage_stock'  => false,
        'stock_status'  => $product->stock == '1' ? 'instock' : 'outofstock',
      ];

      if (!empty($wcCategories)) {
        $payload['categories'] = $wcCategories;
      }

      if ($img = $this->wcImageUrl($product->thumbnail_img)) {
        $payload['images'] = [['src' => $img]];
      }

      $wcResponse = $wc->createProduct($payload);
      if (isset($wcResponse['id'])) {
        $product->woocommerce_id = $wcResponse['id'];
        $product->save();
      } else {
        Log::error('Woo createProduct unexpected response', [
          'payload'  => $payload,
          'response' => $wcResponse,
        ]);
      }
    } catch (\Throwable $e) {
      Log::error('WooCommerce Create Product Exception', ['error' => $e->getMessage()]);
    }

    flash(translate('Product created and synced with WooCommerce!'))->success();
    return redirect()->route('product.index');
  }

  public function show($id)
  {
    $product = Product::with([
      'variations.combinations',
    ])->findOrFail($id);

    return view('backend.product.products.show', compact('product'));
  }

  public function edit(Request $request, $id)
  {
    $product = Product::with(['product_categories', 'categories'])
      ->findOrFail($id);

    $categories = Category::with('childrenCategories')
      ->where('parent_id', 0)
      ->get();

    $selected_category_ids = $product->product_categories
      ? $product->product_categories->pluck('category_id')->toArray()
      : [];

    // $lang is still passed by route but not used in view anymore
    $lang = $request->lang;

    return view('backend.product.products.edit', compact(
      'product',
      'lang',
      'categories',
      'selected_category_ids',
    ));
  }

  public function update(Request $request, $id, WooCommerceService $wc)
  {
    $request->validate([
      'name'  => ['required', 'string', 'max:200'],
      'price' => ['required', 'numeric', 'min:0'],
      'stock' => ['required', 'in:0,1'],
    ]);

    $product = Product::findOrFail($id);
    $product->name          = $request->name;
    $product->unit          = $request->unit;
    $product->description   = $request->description;
    $product->min_qty       = 1;
    $product->max_qty       = 0;
    $product->thumbnail_img = $request->thumbnail_img;
    $product->published     = (string) $request->status === '1' ? 1 : 0;
    $product->lowest_price  = $request->price;
    $product->highest_price = $request->price;
    $product->stock         = $request->stock;
    $product->slug          = $request->slug
      ? Str::slug($request->slug, '-')
      : Str::slug($request->name, '-');

    $product->save();

    // categories
    if ($request->has('category_ids')) {
      $ids = collect($request->category_ids)
        ->map(fn($id) => Category::find($id)?->id)
        ->filter()
        ->values()
        ->all();
      $product->categories()->sync($ids);
    } else {
      $product->categories()->detach();
    }

    // Woo sync
    try {
      $wcCategories = [];
      foreach ($product->categories as $category) {
        $wcCat = $wc->createOrGetCategory($category->name);
        if (is_array($wcCat) && isset($wcCat['id'])) {
          $wcCategories[] = ['id' => $wcCat['id']];
        }
      }

      $payload = [
        'name'          => $product->name,
        'status'        => ((int) $product->published === 1 ? 'publish' : 'draft'),
        'regular_price' => (string) $product->lowest_price,
        'description'   => $product->description,
        'manage_stock'  => false,
        'stock_status'  => $product->stock == '1' ? 'instock' : 'outofstock',
      ];

      if (!empty($wcCategories)) {
        $payload['categories'] = $wcCategories;
      }

      if ($img = $this->wcImageUrl($product->thumbnail_img)) {
        $payload['images'] = [['src' => $img]];
      }

      if ($product->woocommerce_id) {
        $wc->updateProduct($product->woocommerce_id, $payload);
      } else {
        $wcResponse = $wc->createProduct(['type' => 'simple'] + $payload);
        if (isset($wcResponse['id'])) {
          $product->woocommerce_id = $wcResponse['id'];
          $product->save();
        } else {
          Log::error('Woo createProduct (from update) unexpected response', [
            'payload'  => $payload,
            'response' => $wcResponse,
          ]);
        }
      }
    } catch (\Throwable $e) {
      Log::error('WooCommerce Update Product Exception', [
        'error'  => $e->getMessage(),
        'woo_id' => $product->woocommerce_id,
      ]);
    }

    flash(translate('Product updated and synced with WooCommerce!'))->success();
    return redirect()->route('product.index');
  }

  public function destroy($id, WooCommerceService $wc)
  {
    $product = Product::findOrFail($id);

    try {
      if ($product->woocommerce_id) {
        $wc->deleteProduct($product->woocommerce_id);
      }
    } catch (\Exception $e) {
      Log::error('WooCommerce Delete Product Exception', ['error' => $e->getMessage()]);
    }

    $product->product_translations()->delete();
    $product->variations()->delete();
    $product->variation_combinations()->delete();
    $product->reviews()->delete();
    $product->product_categories()->delete();
    $product->offers()->delete();
    $product->wishlists()->delete();
    $product->attributes()->delete();
    $product->attribute_values()->delete();
    $product->taxes()->delete();
    $product->delete();

    flash(translate('Product deleted successfully!'))->success();
    return redirect()->route('product.index');
  }

  public function duplicate(Request $request, $id)
  {
    $product     = Product::find($id);
    $product_new = $product->replicate();
    $product_new->slug = Str::slug($product_new->name, '-') . '-' . strtolower(Str::random(5));

    if ($product_new->save()) {
      foreach ($product->variations as $variation) {
        $p_variation               = new ProductVariation;
        $p_variation->product_id   = $product_new->id;
        $p_variation->code         = $variation->code;
        $p_variation->price        = $variation->price;
        $p_variation->stock        = $variation->stock;
        $p_variation->sku          = $variation->sku;
        $p_variation->img          = $variation->img;
        $p_variation->save();

        foreach ($variation->combinations as $combination) {
          $p_variation_comb                         = new ProductVariationCombination;
          $p_variation_comb->product_id             = $product_new->id;
          $p_variation_comb->product_variation_id   = $p_variation->id;
          $p_variation_comb->attribute_id           = $combination->attribute_id;
          $p_variation_comb->attribute_value_id     = $combination->attribute_value_id;
          $p_variation_comb->save();
        }
      }

      foreach ($product->attributes as $attribute) {
        $p_attribute               = new ProductAttribute;
        $p_attribute->product_id   = $product_new->id;
        $p_attribute->attribute_id = $attribute->attribute_id;
        $p_attribute->save();
      }

      foreach ($product->attribute_values as $attribute_value) {
        $p_attr_value                          = new ProductAttributeValue;
        $p_attr_value->product_id              = $product_new->id;
        $p_attr_value->attribute_id            = $attribute_value->attribute_id;
        $p_attr_value->attribute_value_id      = $attribute_value->attribute_value_id;
        $p_attr_value->save();
      }

      foreach ($product->product_translations as $translation) {
        $product_translation              = new ProductTranslation;
        $product_translation->product_id  = $product_new->id;
        $product_translation->name        = $translation->name;
        $product_translation->unit        = $translation->unit;
        $product_translation->description = $translation->description;
        $product_translation->lang        = $translation->lang;
        $product_translation->save();
      }

      foreach ($product->product_categories as $category) {
        $p_category              = new ProductCategory;
        $p_category->product_id  = $product_new->id;
        $p_category->category_id = $category->category_id;
        $p_category->save();
      }

      foreach ($product->taxes as $tax) {
        $p_tax            = new ProductTax;
        $p_tax->product_id = $product_new->id;
        $p_tax->tax_id     = $tax->tax_id;
        $p_tax->tax        = $tax->tax;
        $p_tax->tax_type   = $tax->tax_type;
        $p_tax->save();
      }

      flash(translate('Product has been duplicated successfully'))->success();
      return redirect()->route('product.index');
    } else {
      flash(translate('Something went wrong'))->error();
      return back();
    }
  }

  public function get_products_by_subcategory(Request $request)
  {
    $products = Product::where('subcategory_id', $request->subcategory_id)->get();
    return $products;
  }

  public function get_products_by_brand(Request $request)
  {
    $products = Product::where('brand_id', $request->brand_id)->get();
    return view('partials.product_select', compact('products'));
  }

  public function updatePublished(Request $request)
  {
    $product            = Product::findOrFail($request->id);
    $product->published = $request->status;
    $product->save();

    cache_clear();

    return 1;
  }

  public function sku_combination(Request $request)
  {
    $option_choices = [];

    if ($request->has('product_options')) {
      $product_options = $request->product_options;
      sort($product_options, SORT_NUMERIC);

      foreach ($product_options as $option) {
        $option_name = 'option_' . $option . '_choices';
        $choices     = [];

        if ($request->has($option_name)) {
          $product_option_values = $request[$option_name];
          sort($product_option_values, SORT_NUMERIC);

          foreach ($product_option_values as $item) {
            $choices[] = $item;
          }
          $option_choices[$option] = $choices;
        }
      }
    }

    $combinations = [[]];
    foreach ($option_choices as $property => $property_values) {
      $tmp = [];
      foreach ($combinations as $combination_item) {
        foreach ($property_values as $property_value) {
          $tmp[] = $combination_item + [$property => $property_value];
        }
      }
      $combinations = $tmp;
    }

    return view('backend.product.products.sku_combinations', compact('combinations'))->render();
  }

  public function new_attribute(Request $request)
  {
    $attributes = Attribute::query();
    if ($request->has('product_attributes')) {
      foreach ($request->product_attributes as $value) {
        if ($value == null) {
          return [
            'count' => -1,
            'view'  => view('backend.product.products.new_attribute', compact('attributes'))->render(),
          ];
        }
      }
      $attributes->whereNotIn('id', array_diff($request->product_attributes, [null]));
    }

    $attributes = $attributes->get();

    return [
      'count' => count($attributes),
      'view'  => view('backend.product.products.new_attribute', compact('attributes'))->render(),
    ];
  }

  public function get_attribute_values(Request $request)
  {
    $attribute_id    = $request->attribute_id;
    $attribute_values = AttributeValue::where('attribute_id', $attribute_id)->get();

    return view('backend.product.products.new_attribute_values', compact('attribute_values', 'attribute_id'));
  }

  public function new_option(Request $request)
  {
    $attributes = Attribute::query();
    if ($request->has('product_options')) {
      foreach ($request->product_options as $value) {
        if ($value == null) {
          return [
            'count' => -1,
            'view'  => view('backend.product.products.new_option', compact('attributes'))->render(),
          ];
        }
      }
      $attributes->whereNotIn('id', array_diff($request->product_options, [null]));
      if (count($request->product_options) === 3) {
        return [
          'count' => -2,
          'view'  => view('backend.product.products.new_option', compact('attributes'))->render(),
        ];
      }
    }

    $attributes = $attributes->get();

    return [
      'count' => count($attributes),
      'view'  => view('backend.product.products.new_option', compact('attributes'))->render(),
    ];
  }

  public function get_option_choices(Request $request)
  {
    $attribute_id    = $request->attribute_id;
    $attribute_values = AttributeValue::where('attribute_id', $attribute_id)->get();

    return view('backend.product.products.new_option_choices', compact('attribute_values', 'attribute_id'));
  }

  public function updateProductApproval(Request $request)
  {
    $product          = Product::findOrFail($request->id);
    $product->approved = $request->approved;
    $shop             = $product->shop;

    if ($shop->user->user_type == 'seller') {
      if (
        $shop->package_invalid_at == null
        || Carbon::now()->diffInDays(Carbon::parse($shop->package_invalid_at), false) < 0
        || $shop->product_upload_limit <= $shop->products()->where('published', 1)->count()
      ) {
        return 0;
      }
    }

    $product->save();
    cache_clear();
    return 1;
  }

  /**
   * Resolve an uploader value (ID/path/URL) into a public absolute URL for Woo.
   */
  private function wcImageUrl($value): ?string
  {
    // Already a full URL
    if (is_string($value) && filter_var($value, FILTER_VALIDATE_URL)) {
      return $value;
    }

    // Numeric ID → uploaded_asset() helper or Upload model
    if (is_numeric($value)) {
      try {
        if (function_exists('uploaded_asset')) {
          $url = uploaded_asset((int) $value);
          if ($url && filter_var($url, FILTER_VALIDATE_URL)) {
            return $url;
          }
        }
        if (class_exists(\App\Models\Upload::class)) {
          $upload = \App\Models\Upload::find((int) $value);
          if ($upload) {
            $path = $upload->file_name ?? $upload->file_original_name ?? null;
            if ($path) {
              $candidate = asset($path);
              if (filter_var($candidate, FILTER_VALIDATE_URL)) {
                return $candidate;
              }
            }
          }
        }
      } catch (\Throwable $e) {
        Log::warning('wcImageUrl resolution failed', [
          'value' => $value,
          'err'   => $e->getMessage(),
        ]);
      }
    }

    // Relative path → make absolute
    if (is_string($value) && !empty($value) && !str_starts_with($value, 'http')) {
      $candidate = asset($value);
      if (filter_var($candidate, FILTER_VALIDATE_URL)) {
        return $candidate;
      }
    }

    return null;
  }
}
