<?php

namespace App\Jobs;

use App\Models\Order;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Log;
use App\Notifications\PaidNotificationAdmin;
use App\Notifications\PaidNotificationCustomer;

class SendOrderNotifications implements ShouldQueue
{
  use InteractsWithQueue, Queueable, SerializesModels;

  public $order;

  public function __construct(Order $order)
  {
    // keep it simple; no heavy loading here
    $this->order = $order;
  }

  public function handle()
  {
    $order = $this->order->fresh(); // make sure we have latest data

    // Admin
    $admin = User::where('user_type', 'admin')->first();
    if ($admin && $this->isValidEmail($admin->email)) {
      $this->safeNotify(function () use ($admin, $order) {
        $admin->notify(new PaidNotificationAdmin($order));
      }, 'admin', $admin->email, $order->id);
    }

    // Reseller
    $user = User::find($order->user_id);
    if ($user && $this->isValidEmail($user->email)) {
      $this->safeNotify(function () use ($user, $order) {
        Notification::route('mail', $user->email)
          ->notify(new PaidNotificationCustomer($order));
      }, 'reseller', $user->email, $order->id);
    }

    // Customer
    if ($this->isValidEmail($order->email)) {
      $this->safeNotify(function () use ($order) {
        Notification::route('mail', $order->email)
          ->notify(new PaidNotificationCustomer($order));
      }, 'customer', $order->email, $order->id);
    }
  }

  private function isValidEmail(?string $email): bool
  {
    return !empty($email) && filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
  }

  private function safeNotify(callable $callback, string $role, string $email, int $orderId): void
  {
    try {
      $callback();
    } catch (\Throwable $e) {
      Log::warning('Failed sending paid notification', [
        'role'     => $role,
        'email'    => $email,
        'order_id' => $orderId,
        'error'    => $e->getMessage(),
      ]);

      // swallow the exception; do NOT rethrow
    }
  }
}
