<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Category extends Model
{
  use SoftDeletes;

  protected $guarded = [];

  protected static function boot()
  {
    parent::boot();

    // Create: normalize and ensure deterministic uniqueness (no random suffix)
    static::creating(function (Category $cat) {
      // If a slug was provided, normalize it; otherwise derive from name
      $base = $cat->slug ? Str::slug($cat->slug, '-') : Str::slug($cat->name, '-');
      $cat->slug = static::uniqueSlug($base);
    });

    // Update: if name changes and slug wasn't explicitly set, regenerate;
    // if slug explicitly changed, normalize and ensure uniqueness.
    static::updating(function (Category $cat) {
      if ($cat->isDirty('slug')) {
        $base = Str::slug((string) $cat->slug, '-');
        $cat->slug = static::uniqueSlug($base, $cat->id);
      } elseif ($cat->isDirty('name')) {
        $base = Str::slug((string) $cat->name, '-');
        $cat->slug = static::uniqueSlug($base, $cat->id);
      }
    });
  }

  /**
   * Deterministic unique slugger: base, base-2, base-3, ...
   * Uses withTrashed() to avoid conflicts with soft-deleted rows.
   */
  public static function uniqueSlug(string $base, ?int $ignoreId = null): string
  {
    $slug = $base !== '' ? $base : 'category';

    $query = static::withTrashed()->where('slug', $slug);
    if ($ignoreId) {
      $query->where('id', '!=', $ignoreId);
    }

    if (!$query->exists()) {
      return $slug;
    }

    $i = 2;
    do {
      $candidate = "{$base}-{$i}";
      $q = static::withTrashed()->where('slug', $candidate);
      if ($ignoreId) {
        $q->where('id', '!=', $ignoreId);
      }
      if (!$q->exists()) {
        return $candidate;
      }
      $i++;
    } while (true);
  }

  // Relations
  public function products()
  {
    return $this->belongsToMany(Product::class, 'product_categories', 'category_id', 'product_id')
      ->withTimestamps();
  }

  public function childrenCategories()
  {
    return $this->hasMany(Category::class, 'parent_id')->with('childrenCategories');
  }

  public function parentCategory()
  {
    return $this->belongsTo(Category::class, 'parent_id');
  }

  public function product_categories()
  {
    return $this->hasMany(ProductCategory::class);
  }

  public function attributes()
  {
    return $this->belongsToMany(Attribute::class);
  }

  public function category_translations()
  {
    return $this->hasMany(\App\Models\CategoryTranslation::class);
  }
}
