<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class CommissionHistory extends Model
{
    protected $fillable = [
        'order_id',
        'user_id',
        'seller_earning',
        'details',
        'source_type',
    ];

    protected $casts = [
        'seller_earning' => 'float',
        'created_at'     => 'datetime',
        'updated_at'     => 'datetime',
    ];

    protected $appends = ['source_label'];

    /* ---------------- Relations ---------------- */

    public function order()
    {
        return $this->belongsTo(Order::class, 'order_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /* ---------------- Accessors ---------------- */

    public function getSourceLabelAttribute(): string
    {
        $type    = strtolower((string) $this->source_type);
        $details = strtolower(trim((string) $this->details));

        // Prefer explicit, normalized source_type.
        switch ($type) {
            case 'order':
            case 'order_sale':
            case 'order_payment':
            case 'order_sale_helper':
                return $this->orderLabel();

            case 'registration':
            case 'package':
                return translate('Affiliate Registration');

            case 'referral_sale':
            case 'referral':
                return translate('Affiliate Sale');

            case 'area_bonus':
            case 'area_sale_bonus':
                return translate('Area Sale Bonus Commission');
        }

        // Infer when source_type is blank or legacy.
        if ($this->order_id || str_contains($details, 'order payment')) {
            return $this->orderLabel();
        }
        if (str_contains($details, 'package payment')) {
            return translate('Affiliate Registration');
        }
        if (str_contains($details, 'referral sale commission')) {
            return translate('Affiliate Sale');
        }
        if (str_contains($details, 'area sale bonus')) {
            return translate('Area Sale Bonus Commission');
        }

        return translate('Other');
    }

    /* ---------------- Helpers ---------------- */

    private function orderLabel(): string
    {
        // Try to show an order reference if available.
        $orderCode = null;

        if ($this->relationLoaded('order') ? $this->order : $this->order()->exists()) {
            $order     = $this->relationLoaded('order') ? $this->order : $this->order()->first();
            $orderCode = $order->code
                ?? optional($order->combined_order)->code
                ?? ($this->order_id ?: null);
        } elseif ($this->order_id) {
            $orderCode = $this->order_id;
        }

        // If you prefer plain label without number, return translate('Customer Order').
        return $orderCode
            ? translate('Sale Commission from Order') . ' #' . $orderCode
            : translate('Sale Commission from Order');
    }
}
