<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Services\Phone;

class Order extends Model
{
  // Choose ONE strategy. Keep fillable; drop guarded.
  protected $fillable = [
    'user_id',
    'combined_order_id',
    'code',
    'name',
    'email',
    'phone',
    'shipping_address',
    'billing_address',
    'shipping_cost',
    'barangay_id',
    'city_id',
    'province_id',
    'region_id',
    'delivery_type',
    'delivery_status',
    'payment_type',
    'payment_status',
    'payment_details',
    'grand_total',
    'seller_earning',
    'commission_percentage',
    'courier_name',
    'tracking_number',
    'tracking_url',
    'ship_to_other',
    'recipient_name',
    'recipient_phone',
    'recipient_address',
  ];

  protected $casts = [
    'ship_to_other'         => 'bool',
    'grand_total'           => 'float',
    'seller_earning'        => 'float',
    'commission_percentage' => 'float',
  ];

  // --- Relations ---
  public function orderDetails()
  {
    return $this->hasMany(OrderDetail::class);
  }

  public function user()
  {
    return $this->belongsTo(User::class, 'user_id');
  }

  public function combined_order()
  {
    return $this->belongsTo(CombinedOrder::class);
  }

  public function commission_histories()
  {
    return $this->hasMany(CommissionHistory::class);
  }

  public function order_updates() // fixed name
  {
    return $this->hasMany(OrderUpdate::class)->latest();
  }

  public function refundRequests()
  {
    return $this->hasMany(RefundRequest::class);
  }

  public function barangay()
  {
    return $this->belongsTo(Barangay::class, 'barangay_id');
  }

  public function city()
  {
    return $this->belongsTo(City::class, 'city_id');
  }

  public function province() // use correct FK name per schema
  {
    return $this->belongsTo(State::class, 'province_id');
  }

  public function region()
  {
    return $this->belongsTo(Region::class, 'region_id');
  }

  public function setPhoneAttribute($value)
  {
    $this->attributes['phone'] = Phone::normalizePhMobile($value) ?? $value;
  }

  public function setRecipientPhoneAttribute($value)
  {
    $this->attributes['recipient_phone'] = Phone::normalizePhMobile($value) ?? null;
  }
}
