<?php

namespace App\Models;

use Illuminate\Notifications\Notifiable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Laravel\Passport\HasApiTokens;
use App\Models\Cart;
use App\Models\Wishlist;
use App\Models\Product;
use App\Models\Order;
use App\Models\Wallet;
use App\Models\Review;
use App\Models\Address;
use Illuminate\Support\Facades\Cache;
use App\Models\SellerPackagePayment;
use App\Models\CommissionHistory;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable implements MustVerifyEmail
{
  use Notifiable, HasApiTokens, HasRoles;

  protected $fillable = [
    'name',
    'email',
    'password',
    'phone',
    'avatar',
    'provider_id',
    'email_verified_at',
    'verification_code',
    'referral_code'
  ];

  protected $hidden = [
    'password',
    'remember_token',
  ];

  public function wishlists()
  {
    return $this->hasMany(Wishlist::class);
  }

  public function role()
  {
    return $this->belongsTo(Role::class);
  }

  public function orders()
  {
    return $this->hasMany(Order::class, 'user_id');
  }

  public function wallets()
  {
    return $this->hasMany(Wallet::class)->orderBy('created_at', 'desc');
  }

  public function carts()
  {
    return $this->hasMany(Cart::class);
  }

  public function reviews()
  {
    return $this->hasMany(Review::class);
  }

  public function addresses()
  {
    return $this->hasMany(Address::class);
  }

  public function chat_thread()
  {
    return $this->hasOne(ChatThread::class);
  }

  public function conversations()
  {
    return $this->hasMany(Conversation::class, 'sender_id', 'id');
  }

  public function affiliate_user()
  {
    return $this->hasOne(AffiliateUser::class);
  }
  public function affiliate_withdraw_request()
  {
    return $this->hasMany(AffiliateWithdrawRequest::class);
  }
  public function affiliate_log()
  {
    return $this->hasMany(AffiliateLog::class);
  }
  public function seller_package()
  {
    return $this->belongsTo(SellerPackage::class, 'package_id');
  }
  public function sellerPackagePayment()
  {
    return $this->hasOne(SellerPackagePayment::class, 'user_id')->with('seller_package');
  }

  public function sellerPackagePayments()
  {
    return $this->hasMany(SellerPackagePayment::class, 'user_id');
  }

  public function commission_histories()
  {
    return $this->hasMany(CommissionHistory::class, 'user_id', 'referred_by')
      ->where('details', 'Package Payment.');
  }
  public function locationAssignments()
  {
    return $this->hasMany(\App\Models\UserLocationAssignment::class);
  }

  public function products()
  {
    return $this->hasMany(Product::class, 'user_id', 'id');
  }

  public function package()
  {
    return $this->belongsTo(SellerPackage::class, 'package_id');
  }

  public function getAssignedNameAttribute(): string
  {
    return Cache::remember("user:{$this->id}:assigned_name", 300, function () {
      $leafId = (int) ($this->seller_assign_location ?? 0);
      $level  = (int) ($this->geographic_level ?? 0);

      // Primary: explicit leaf+level on users table
      if ($leafId > 0 && $level > 0) {
        switch ($level) {
          case 1: // barangay
            if ($b = Barangay::select('id', 'name', 'city_id')->find($leafId)) {
              $c  = City::select('id', 'name', 'state_id')->find($b->city_id);
              $st = $c ? State::select('id', 'name', 'region_id')->find($c->state_id) : null;
              return trim($b->name
                . ($c ? ', '  . $c->name  : '')
                . ($st ? ', ' . $st->name : ''));
            }
            break;

          case 2: // city
            if ($c = City::select('id', 'name', 'state_id')->find($leafId)) {
              $st = State::select('id', 'name', 'region_id')->find($c->state_id);
              return trim($c->name . ($st ? ', ' . $st->name : ''));
            }
            break;

          case 3: // state/province
            if ($st = State::select('id', 'name', 'region_id')->find($leafId)) {
              $r = Region::select('id', 'name')->find($st->region_id);
              return trim($st->name . ($r ? ', ' . $r->name : ''));
            }
            break;

          case 4: // region (seller_assign is VARCHAR historically)
            if ($r = Region::select('id', 'name')->find($leafId)) {
              return $r->name;
            }
            break;
        }
      }

      // Fallback: legacy records where geo tables store seller_assign = user_id
      if ($b = Barangay::select('id', 'name', 'city_id')->where('seller_assign', $this->id)->first()) {
        $c  = City::select('id', 'name', 'state_id')->find($b->city_id);
        $st = $c ? State::select('id', 'name', 'region_id')->find($c->state_id) : null;
        return trim($b->name
          . ($c ? ', '  . $c->name  : '')
          . ($st ? ', ' . $st->name : ''));
      }

      if ($c = City::select('id', 'name', 'state_id')->where('seller_assign', $this->id)->first()) {
        $st = State::select('id', 'name', 'region_id')->find($c->state_id);
        return trim($c->name . ($st ? ', ' . $st->name : ''));
      }

      if ($st = State::select('id', 'name', 'region_id')->where('seller_assign', $this->id)->first()) {
        $r = Region::select('id', 'name')->find($st->region_id);
        return trim($st->name . ($r ? ', ' . $r->name : ''));
      }

      // regions.seller_assign is VARCHAR
      if ($r = Region::select('id', 'name')->where('seller_assign', (string) $this->id)->first()) {
        return $r->name;
      }

      return '—';
    });
  }
}
