<?php

namespace App\Addons\Multivendor\Http\Controllers\Admin;

use Hash;
use Artisan;
use App\Models\City;
use App\Models\Shop;
use App\Models\User;
use App\Models\Order;
use App\Models\State;
use App\Models\Region;
use App\Models\Product;
use App\Models\Barangay;
use App\Models\OrderDetail;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\SellerPackagePayment;
use App\Notifications\EmailVerificationNotification;
use App\Notifications\SellerAccountApprovedNotification;
use MehediIitdu\CoreComponentRepository\CoreComponentRepository;

class SellerController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_seller_products'])->only('seller_products');
    $this->middleware(['permission:show_seller_orders'])->only('seller_orders');
    $this->middleware(['permission:show_sellers'])->only('all_sellers');
  }

  // Admin Panel
  // public function all_sellers(Request $request)
  // {
  //   $admin = User::where('user_type', 'admin')->first();

  //   $sort_search = null;
  //   $approved = null;
  //   $shops = Shop::withCount('products')->with(['user', 'seller_package'])->where('id', '!=', $admin->shop_id);
  //   //$shops = Shop::with(['user', 'seller_package'])->where('id', '!=', $admin->shop_id);


  //   if ($request->has('approved_status') && $request->approved_status != null) {
  //     $approved = $request->approved_status;
  //     $shops = $shops->where('approval', $approved);
  //   }

  //   if ($request->has('search') && $request->search != null) {
  //     $sort_search = $request->search;
  //     $shops = $shops->where('name', 'like', '%' . $sort_search . '%')
  //       ->orWhere('phone', 'like', '%' . $sort_search . '%')
  //       ->orWhereHas('user', function ($query) use ($sort_search) {
  //         $query->where('name', 'like', '%' . $sort_search . '%');
  //       });
  //   }

  //   $shops = $shops->latest()->paginate(15);
  //   return view('addon:multivendor::admin.sellers.index', compact('shops', 'sort_search', 'approved'));
  // }

  public function all_sellers(Request $request)
  {
    $sort_search = null;
    $approved    = null;

    // Base: only users who are sellers
    $sellers = User::query()
      ->where('user_type', 'seller')
      // ->withCount('products')   // re-enable ONLY if User::products() uses the correct FK (e.g., user_id)
      ->with(['seller_package', 'sellerPackagePayment']); // eager load what's shown in the table

    // Filter: approval status
    if ($request->filled('approved_status')) {
      $approved = (int) $request->approved_status;
      $sellers->where('approval', $approved);
    }

    // Filter: search across name/phone/email
    if ($request->filled('search')) {
      $sort_search = trim($request->search);
      $sellers->where(function ($q) use ($sort_search) {
        $q->where('name', 'like', "%{$sort_search}%")
          ->orWhere('phone', 'like', "%{$sort_search}%")
          ->orWhere('email', 'like', "%{$sort_search}%");
      });
    }

    $sellers = $sellers
      ->latest('id')
      ->paginate(15)
      ->appends($request->query());

    return view('addon:multivendor::admin.sellers.index', [
      'sellers'     => $sellers,  // <-- update the blade to use $sellers
      'sort_search' => $sort_search,
      'approved'    => $approved,
    ]);
  }



  public function seller_create()
  {
    return view('addon:multivendor::admin.sellers.create');
  }

  public function seller_store(Request $request)
  {
    if (User::where('email', $request->email)->first() != null) {
      flash(translate('Email already exists!'))->error();
      return back();
    }
    if ($request->password == $request->confirm_password) {
      $user = new User;
      $user->name = $request->name;
      $user->email = $request->email;
      $user->phone = $request->phone;
      $user->user_type = "seller";
      $user->password = Hash::make($request->password);

      if ($user->save()) {
        if (get_setting('email_verification') != 1) {
          $user->email_verified_at = date('Y-m-d H:m:s');
        } else {
          $user->notify(new EmailVerificationNotification());
        }
        $user->save();

        $shop = new Shop;
        $shop->user_id = $user->id;
        $shop->name = $request->shop_name;
        $shop->slug = preg_replace('/\s+/', '-', $request->shop_name) . '-' . $shop->id;
        $shop->save();

        flash(translate('Seller has been added successfully'))->success();
        return redirect()->route('admin.all_sellers');
      }
      flash(translate('Something went wrong'))->error();
      return back();
    } else {
      flash("Password and confirm password didn't match")->warning();
      return back();
    }
  }



  public function seller_edit($id)
  {
    $seller = User::findOrFail(decrypt($id));
    return view('addon:multivendor::admin.sellers.edit', compact('seller'));
  }

  public function seller_update(Request $request)
  {
    $user = User::findOrFail($request->seller_id);
    $user->name = $request->name;
    $user->email = $request->email;
    if (User::where('id', '!=', $user->id)->where('email', $request->email)->first() == null) {
      if (strlen($request->password) > 0) {
        $user->password = Hash::make($request->password);
      }
      if ($user->save()) {
        flash(translate('Seller has been updated successfully'))->success();
        return redirect()->route('admin.all_sellers');
      }

      flash(translate('Something went wrong'))->error();
      return back();
    } else {
      flash(translate('Email Already Exists!'))->error();
      return back();
    }
  }

  public function seller_destroy($id)
  {
    // Retrieve the user by ID
    $user = User::findOrFail($id);

    // Retrieve approval status using the SellerPackagePayment model
    $approvalStatus = SellerPackagePayment::where('user_id', $user->id)
      ->value('approval');

    if ($approvalStatus == 0) {
      // Retrieve the user's shop_id (assuming the user has a shop linked)
      $shopId = Shop::where('user_id', $user->id)->value('id');

      // Update seller_assign field if it matches the shop_id in relevant tables
      Barangay::where('seller_assign', $shopId)->update(['seller_assign' => null]);
      City::where('seller_assign', $shopId)->update(['seller_assign' => null]);
      State::where('seller_assign', $shopId)->update(['seller_assign' => null]);
      Region::where('seller_assign', $shopId)->update(['seller_assign' => null]);

      // Delete the user after updating related tables
      Shop::where('user_id', $user->id)->delete();
      if (User::destroy($id)) {
        flash(translate('Seller has been deleted successfully'))->success();
        return redirect()->route('admin.all_sellers');
      } else {
        flash(translate('Something went wrong'))->error();
        return back();
      }
    } else {
      // If approval is not 0, set user approval to 0 in shops table
      Shop::where('user_id', $user->id)->update(['approval' => 0]);

      flash(translate('User approval has been updated'))->success();
      return redirect()->route('admin.all_sellers');
    }
  }

  // public function update_seller_approval(Request $request)
  // {
  //   $request->validate([
  //     'id'     => 'required|integer',
  //     'status' => 'required|in:0,1',
  //   ]);

  //   $userId    = (int) $request->id;
  //   $turnOn    = (int) $request->status === 1;

  //   $user = \App\Models\User::findOrFail($userId);

  //   $wasApproved = (int) ($user->approval ?? 0) === 1;

  //   \DB::beginTransaction();
  //   try {
  //     // Set approval
  //     $user->approval = $turnOn ? 1 : 0;

  //     if ($turnOn) {
  //       // Find the most recent payment for this user (has seller_assign_location id set during signup)
  //       $payment = \App\Models\SellerPackagePayment::with('seller_package')
  //         ->where('user_id', $user->id)
  //         ->latest('id')
  //         ->first();

  //       // Derive level and location id
  //       $level = (int) optional($payment?->seller_package)->geographic_level ?? (int) ($user->geographic_level ?? 0);
  //       $locId = (int) ($payment->seller_assign_location ?? 0); // <-- this should already be the numeric ID

  //       // Claim the location (only if level>0 and we have a location id)
  //       if ($level > 0 && $locId > 0) {
  //         switch ($level) {
  //           case 1: { // barangay
  //               $geo = \App\Models\Barangay::whereKey($locId)->lockForUpdate()->firstOrFail();
  //               if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $user->id) {
  //                 throw new \RuntimeException('This barangay is already assigned to another user.');
  //               }
  //               $geo->seller_assign = $user->id;
  //               $geo->save();
  //               break;
  //             }
  //           case 2: { // city
  //               $geo = \App\Models\City::whereKey($locId)->lockForUpdate()->firstOrFail();
  //               if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $user->id) {
  //                 throw new \RuntimeException('This city is already assigned to another user.');
  //               }
  //               $geo->seller_assign = $user->id;
  //               $geo->save();
  //               break;
  //             }
  //           case 3: { // state
  //               $geo = \App\Models\State::whereKey($locId)->lockForUpdate()->firstOrFail();
  //               if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $user->id) {
  //                 throw new \RuntimeException('This province is already assigned to another user.');
  //               }
  //               $geo->seller_assign = $user->id;
  //               $geo->save();
  //               break;
  //             }
  //           case 4: { // region
  //               $geo = \App\Models\Region::whereKey($locId)->lockForUpdate()->firstOrFail();
  //               if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $user->id) {
  //                 throw new \RuntimeException('This region is already assigned to another user.');
  //               }
  //               $geo->seller_assign = $user->id;
  //               $geo->save();
  //               break;
  //             }
  //         }

  //         // Mirror onto user
  //         $user->seller_assign_location = $locId; // keep ID on users table
  //         if (property_exists($user, 'geographic_level')) {
  //           $user->geographic_level = $level;    // if you added this column
  //         }
  //       }
  //     }

  //     $user->save();
  //     \DB::commit();

  //     // Fire notifications AFTER commit (only first time approval)
  //     if ($turnOn && !$wasApproved) {
  //       try {
  //         // Replace with your actual notification class (or keep try/catch if missing)
  //         if (class_exists(SellerAccountApprovedNotification::class)) {
  //           $user->notify(new \App\Notifications\SellerAccountApprovedNotification($user));
  //         }
  //       } catch (\Throwable $e) {
  //         // Optional: log error
  //       }
  //     }

  //     return response()->json(1);
  //   } catch (\Throwable $e) {
  //     \DB::rollBack();
  //     // Optional: \Log::error($e->getMessage(), ['user_id' => $userId]);
  //     return response()->json(0);
  //   }
  // }

  public function update_seller_approval(Request $request)
  {
    $request->validate([
      'id'     => 'required|integer',
      'status' => 'required|in:0,1',
    ]);

    $userId    = (int) $request->id;
    $turnOn    = (int) $request->status === 1;

    $user = \App\Models\User::findOrFail($userId);

    // ⛔ Block turning ON if package needs payment and payment isn't approved yet
    if ($turnOn) {
      $payment = \App\Models\SellerPackagePayment::with('seller_package')
        ->where('user_id', $user->id)
        ->latest('id')
        ->first();

      $level           = (int) optional($payment?->seller_package)->geographic_level
        ?? (int) ($user->geographic_level ?? 0);
      $paymentApproved = (int) ($payment->approval ?? 0) === 1;

      if ($level > 0 && !$paymentApproved) {
        // Frontend expects 1 on success, 0 on error
        return response()->json(0);
      }
    }

    $wasApproved = (int) ($user->approval ?? 0) === 1;

    \DB::beginTransaction();
    try {
      // Set approval
      $user->approval = $turnOn ? 1 : 0;

      if ($turnOn) {
        // Find the most recent payment for this user (has seller_assign_location id set during signup)
        $payment = \App\Models\SellerPackagePayment::with('seller_package')
          ->where('user_id', $user->id)
          ->latest('id')
          ->first();

        // Derive level and location id
        $level = (int) optional($payment?->seller_package)->geographic_level ?? (int) ($user->geographic_level ?? 0);
        $locId = (int) ($payment->seller_assign_location ?? 0); // <-- this should already be the numeric ID

        // Claim the location (only if level>0 and we have a location id)
        if ($level > 0 && $locId > 0) {
          switch ($level) {
            case 1: { // barangay
                $geo = \App\Models\Barangay::whereKey($locId)->lockForUpdate()->firstOrFail();
                if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $user->id) {
                  throw new \RuntimeException('This barangay is already assigned to another user.');
                }
                $geo->seller_assign = $user->id;
                $geo->save();
                break;
              }
            case 2: { // city
                $geo = \App\Models\City::whereKey($locId)->lockForUpdate()->firstOrFail();
                if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $user->id) {
                  throw new \RuntimeException('This city is already assigned to another user.');
                }
                $geo->seller_assign = $user->id;
                $geo->save();
                break;
              }
            case 3: { // state
                $geo = \App\Models\State::whereKey($locId)->lockForUpdate()->firstOrFail();
                if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $user->id) {
                  throw new \RuntimeException('This province is already assigned to another user.');
                }
                $geo->seller_assign = $user->id;
                $geo->save();
                break;
              }
            case 4: { // region
                $geo = \App\Models\Region::whereKey($locId)->lockForUpdate()->firstOrFail();
                if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $user->id) {
                  throw new \RuntimeException('This region is already assigned to another user.');
                }
                $geo->seller_assign = $user->id;
                $geo->save();
                break;
              }
          }

          // Mirror onto user
          $user->seller_assign_location = $locId; // keep ID on users table
          if (property_exists($user, 'geographic_level')) {
            $user->geographic_level = $level;    // if you added this column
          }
        }
      }

      $user->save();
      \DB::commit();

      // Fire notifications AFTER commit (only first time approval)
      if ($turnOn && !$wasApproved) {
        try {
          if (class_exists(\App\Notifications\SellerAccountApprovedNotification::class)) {
            $user->notify(new \App\Notifications\SellerAccountApprovedNotification($user));
          }
        } catch (\Throwable $e) {
          // Optional: log error
        }
      }

      return response()->json(1);
    } catch (\Throwable $e) {
      \DB::rollBack();
      return response()->json(0);
    }
  }

  public function update_shop_publish(Request $request)
  {
    $shop = Shop::findOrFail($request->id);
    $shop->published = $request->status;

    cache_clear();

    if ($shop->save()) {
      return 1;
    }
    return 0;
  }


  public function profile_modal(Request $request)
  {
    $seller = \App\Models\User::with(['seller_package', 'sellerPackagePayment'])
      ->findOrFail($request->id);

    return view('addon:multivendor::admin.sellers.profile_modal', compact('seller'));
  }


  public function payment_modal(Request $request)
  {
    $seller = User::findOrFail($request->id);
    return view('addon:multivendor::admin.seller_payouts.payment_modal', compact('seller'));
  }


  public function seller_products(Request $request)
  {
    $col_name = null;
    $query = null;
    $sort_search = null;
    $user_id = null;

    $admin = User::where('user_type', 'admin')->first();
    $products = Product::orderBy('created_at', 'desc')->where('user_id', '!=', $admin->user->id);

    if ($request->user_id != null) {
      $user_id = $request->user_id;
      $products = $products->where('user_id', $user_id);
    }

    if ($request->search != null) {
      $products = $products->where('name', 'like', '%' . $request->search . '%');
      $sort_search = $request->search;
    }
    if ($request->type != null) {
      $var = explode(",", $request->type);
      $col_name = $var[0];
      $query = $var[1];
      $products = $products->orderBy($col_name, $query);
      $sort_type = $request->type;
    }

    $products = $products->paginate(15);
    $type = 'All';

    return view('addon:multivendor::admin.seller_products', compact('products', 'type', 'col_name', 'query', 'sort_search', 'user_id'));
  }

  public function seller_orders(Request $request)
  {

    $payment_status = null;
    $delivery_status = null;
    $sort_search = null;

    $admin = User::where('user_type', 'admin')->first();
    $orders = Order::with(['combined_order', 'user'])
      ->whereNotNull('code');

    if ($request->has('search') && $request->search != null) {
      $sort_search = $request->search;
      $orders = $orders->whereHas('combined_order', function ($query) use ($sort_search) {
        $query->where('code', 'like', '%' . $sort_search . '%');
      });
    }

    if ($request->payment_status != null) {
      $orders = $orders->where('payment_status', $request->payment_status);
      $payment_status = $request->payment_status;
    }

    if ($request->delivery_status != null) {
      $orders = $orders->where('delivery_status', $request->delivery_status);
      $delivery_status = $request->delivery_status;
    }

    $orders = $orders->latest()->paginate(15);
    return view('addon:multivendor::admin.orders.seller_orders', compact('orders', 'payment_status', 'delivery_status', 'sort_search'));
  }

  public function seller_order_show($id)
  {
    // $order = Order::with(['orderDetails.product', 'orderDetails.variation.combinations'])->findOrFail($id);
    $order = Order::with(['orderDetails.product', 'orderDetails.variation.combinations', 'order_updates.user'])->findOrFail($id);


    // Get the shop ID from the order
    $userId = $order->user_id;

    // Retrieve the shop using the shop ID from the order
    $user = User::findOrFail($userId);

    // Access the seller package and retrieve the discount
    $discount = $user->seller_package->prod_discount;

    return view('backend.orders.seller_orders_show', compact('order', 'discount'));
  }
}
