<?php

namespace App\Addons\Multivendor\Http\Controllers\Admin;

use Artisan;
use Carbon\Carbon;
use App\Models\City;
use App\Models\Shop;
use App\Models\User;
use App\Models\State;
use App\Models\Region;
use App\Models\Barangay;
use App\Models\UserLocationAssignment;
use Illuminate\Http\Request;
use App\Models\SellerPackage;
use App\Models\CommissionHistory;
use App\Http\Controllers\Controller;
use App\Models\SellerPackagePayment;
use App\Models\SellerPackageTranslation;
use App\Notifications\SellerReferralNewUserPaymentUploadNotification;
use App\Notifications\SellerReferralOldUserPaymentUploadNotification;

class SellerPackageController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_seller_packages'])->only('index');
    $this->middleware(['permission:show_seller_package_payments'])->only('package_purchase_history');
  }

  /**
   * Display a listing of the resource.
   *
   * @return \Illuminate\Http\Response
   */
  public function index()
  {
    $seller_packages = SellerPackage::all();
    return view('addon:multivendor::admin.seller_packages.index', compact('seller_packages'));
  }

  /**
   * Show the form for creating a new resource.
   *
   * @return \Illuminate\Http\Response
   */
  public function create()
  {
    return view('addon:multivendor::admin.seller_packages.create');
  }

  public function store(Request $request)
  {
    $seller_package = new SellerPackage;
    $seller_package->name = $request->name;
    $seller_package->name = $request->name;
    $seller_package->geographic_level = $request->geographic_level;

    $seller_package->commission = $request->commission;
    $seller_package->commission_type = $request->commission_type;

    $seller_package->ref_commission = $request->ref_commission;
    $seller_package->ref_commission_type = $request->ref_commission_type;

    $seller_package->prod_discount = $request->prod_discount;
    $seller_package->prod_discount_type = $request->prod_discount_type;

    $seller_package->ref_sale_commission = $request->ref_sale_commission;
    $seller_package->ref_sale_commission_type = $request->ref_sale_commission_type;

    $seller_package->area_sale_bonus = $request->area_sale_bonus;
    $seller_package->area_sale_bonus_type = $request->area_sale_bonus_type;

    $seller_package->logo = $request->logo;

    if ($seller_package->save()) {
      $seller_package_translation = SellerPackageTranslation::firstOrNew([
        'lang' => env('DEFAULT_LANGUAGE'),
        'seller_package_id' => $seller_package->id
      ]);
      $seller_package_translation->name = $request->name;
      $seller_package_translation->save();

      flash(translate('Package has been added successfully'))->success();
      return redirect()->route('admin.seller_packages.index');
    } else {
      flash(translate('Something went wrong'))->error();
      return back();
    }
  }


  /**
   * Display the specified resource.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function show($id)
  {
    //
  }

  /**
   * Show the form for editing the specified resource.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function edit(Request $request, $id)
  {
    $lang = $request->lang;
    $seller_package = SellerPackage::findOrFail($id);
    return view('addon:multivendor::admin.seller_packages.edit', compact('seller_package', 'lang'));
  }

  public function update(Request $request, $id)
  {
    $seller_package = SellerPackage::findOrFail($id);
    $seller_package->name = $request->name;
    $seller_package->geographic_level = $request->geographic_level;
    $seller_package->amount = $request->amount;
    $seller_package->commission = $request->commission;
    $seller_package->commission_type = $request->commission_type;
    $seller_package->ref_commission = $request->ref_commission;
    $seller_package->ref_commission_type = $request->ref_commission_type;
    $seller_package->prod_discount = $request->prod_discount;
    $seller_package->prod_discount_type = $request->prod_discount_type;
    $seller_package->ref_sale_commission = $request->ref_sale_commission;
    $seller_package->ref_sale_commission_type = $request->ref_sale_commission_type;
    $seller_package->area_sale_bonus = $request->area_sale_bonus;
    $seller_package->area_sale_bonus_type = $request->area_sale_bonus_type;
    $seller_package->logo = $request->logo;

    if ($seller_package->save()) {
      flash(translate('Package has been updated successfully'))->success();
      return redirect()->route('admin.seller_packages.index');
    } else {
      flash(translate('Something went wrong'))->error();
      return back();
    }
  }


  /**
   * Remove the specified resource from storage.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function destroy($id)
  {
    $seller_package = SellerPackage::findOrFail($id);
    foreach ($seller_package->seller_package_translations as $key => $seller_package_translation) {
      $seller_package_translation->delete();
    }
    SellerPackage::destroy($id);
    flash(translate('Package has been deleted successfully'))->success();
    return redirect()->route('admin.seller_packages.index');
  }

  public function package_purchase_history()
  {
    // $package_payments = SellerPackagePayment::latest()->paginate(20);
    // return view('addon:multivendor::admin.seller_packages.payment_history', compact('package_payments'));
    $package_payments = SellerPackagePayment::with('seller_package')->latest()->paginate(20);
    return view('addon:multivendor::admin.seller_packages.payment_history', compact('package_payments'));
  }

  public function package_payment_modal(Request $request)
  {
    $seller_package_payment = SellerPackagePayment::where('id', $request->id)->first();
    return view('addon:multivendor::admin.seller_packages.package_payment_modal', compact('seller_package_payment'));
  }

  // public function check_seller_package_validation(Request $request)
  // {
  //   $admin = User::where('user_type', 'admin')->first();

  //   cache_clear();

  //   foreach (Shop::where('id', '!=', $admin->shop_id)->with(['products', 'seller_package'])->get() as $shop) {
  //     if (seller_package_validity_check($shop->seller_package, $shop->package_invalid_at) != 'active') {
  //       foreach ($shop->products as $product) {
  //         $product->published = 0;
  //         $product->save();
  //       }
  //       $shop->seller_package_id = null;
  //       $shop->package_invalid_at = null;
  //       $shop->published = 0;
  //       $shop->save();
  //     }
  //   }
  // }

  private function labelForLevel(int $level): string
  {
    return match ($level) {
      1 => 'Reseller',          // barangay
      2 => 'City Distributor',  // city
      3 => 'Provincial Dist.',  // state
      4 => 'Regional Dist.',    // region
      default => 'Affiliate',
    };
  }

  public function package_payment_approval(Request $request)
  {
    $payment = \App\Models\SellerPackagePayment::findOrFail($request->id);

    // Already approved? Nothing to do.
    if ((int) $payment->approval === 1) {
      flash(translate('Payment already approved.'))->warning();
      return back();
    }

    $user          = $payment->user;                     // buyer
    $userId        = $user->id;
    $shop          = $user->shop;                        // keep legacy usage if you still need it
    $sellerPackage = \App\Models\SellerPackage::findOrFail($payment->seller_package_id);

    // Where to claim
    $level = (int) $payment->geographic_level;           // 0..4 (leave as-is per your current schema)
    $locId = (int) $payment->seller_assign_location;     // numeric id of barangay/city/state/region

    // Label helper still available in this controller
    $sellerTypeLabel = $this->labelForLevel($level);

    try {
      \DB::beginTransaction();

      // 1) Claim the geo location on approval (first approved wins)
      if ($level > 0 && $locId > 0) {
        switch ($level) {
          case 1: // barangay
            $geo = \App\Models\Barangay::whereKey($locId)->lockForUpdate()->firstOrFail();
            if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $userId) {
              \DB::rollBack();
              flash(translate('This barangay is already assigned to another user.'))->error();
              return back();
            }
            $geo->seller_assign = $userId;
            $geo->save();
            break;

          case 2: // city
            $geo = \App\Models\City::whereKey($locId)->lockForUpdate()->firstOrFail();
            if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $userId) {
              \DB::rollBack();
              flash(translate('This city is already assigned to another user.'))->error();
              return back();
            }
            $geo->seller_assign = $userId;
            $geo->save();
            break;

          case 3: // state
            $geo = \App\Models\State::whereKey($locId)->lockForUpdate()->firstOrFail();
            if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $userId) {
              \DB::rollBack();
              flash(translate('This province is already assigned to another user.'))->error();
              return back();
            }
            $geo->seller_assign = $userId;
            $geo->save();
            break;

          case 4: // region
            $geo = \App\Models\Region::whereKey($locId)->lockForUpdate()->firstOrFail();
            if (!is_null($geo->seller_assign) && (int) $geo->seller_assign !== (int) $userId) {
              \DB::rollBack();
              flash(translate('This region is already assigned to another user.'))->error();
              return back();
            }
            $geo->seller_assign = $userId;
            $geo->save();
            break;
        }
      }

      // 2) Update the user's PRIMARY seller_type to the *package name*
      $user->seller_type = $sellerPackage->name;
      $user->save();

      // 3) Persist a location assignment row ONLY for levels 2..4
      if ($level >= 2 && $locId > 0) {
        \App\Models\UserLocationAssignment::updateOrCreate(
          [
            'user_id'          => $userId,
            'geographic_level' => $level,
            'location_id'      => $locId,
          ],
          [
            'seller_type' => $sellerPackage->name,
            'is_active'   => true,
          ]
        );
      }
      // level 1: barangay claimed above; no UserLocationAssignment row

      // 4) Approve payment
      $payment->approval = 1;
      $payment->save();

      $user->package_id  = $sellerPackage->id;
      $user->seller_type = $sellerPackage->name;
      $user->save();

      // 5) Commission logic (always credit referrer if exists)
      $referrerUser = \App\Models\User::find($user->referred_by);
      $referrerPackage = $referrerUser
        ? \App\Models\SellerPackage::find($referrerUser->package_id)
        : null;

      $sellerEarning = $referrerPackage
        ? ($sellerPackage->amount * $referrerPackage->ref_commission / 100)
        : 0;

      // 6) Optional: keep using Shop for package/commission status (unchanged)
      if ($shop) {
        $shop->commission         = $sellerPackage->commission;
        $shop->seller_package_id  = $payment->seller_package_id;
        $shop->approval           = 1;
        $shop->save();
      }

      // 7) Credit referrer (user-based)
      if ($referrerUser && $sellerEarning > 0) {
        $commissionHistory = new \App\Models\CommissionHistory();
        $commissionHistory->user_id        = $referrerUser->id;
        $commissionHistory->seller_earning = $sellerEarning;
        $commissionHistory->details        = 'Package Payment';
        $commissionHistory->source_type    = 'registration';
        $commissionHistory->save();

        // Update referrer's running balance on users table
        $referrerUser->current_balance = (float) ($referrerUser->current_balance ?? 0) + $sellerEarning;
        $referrerUser->save();
      }

      // 8) Notifications (unchanged)
      if ((int) $payment->self_purchased === 1) {
        $user->notify(new \App\Notifications\SellerReferralOldUserPaymentUploadNotification($shop));
      } else {
        $user->notify(new \App\Notifications\SellerReferralNewUserPaymentUploadNotification($shop));
      }

      \DB::commit();
      flash(translate('Payment approved'))->success();
      return back();
    } catch (\Throwable $e) {
      \DB::rollBack();
      \Log::error('Payment approval failed', ['error' => $e->getMessage(), 'payment_id' => $payment->id]);
      flash(translate('Unable to approve this payment.'))->error();
      return back();
    }
  }
}
