<?php

namespace App\Addons\Multivendor\Http\Controllers\Seller;

use Session;
use App\Models\User;
use App\Models\Order;
use App\Models\Product;
use App\Models\OrderDetail;
use App\Models\OrderUpdate;
use Illuminate\Http\Request;
use App\Models\CombinedOrder;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Http\Resources\PosProductCollection;
use Illuminate\Support\Collection;

class PlaceOrderController extends Controller
{
  /* ---------- Entry page ---------- */
  public function place_order(Request $request)
  {
    $user = auth()->user();
    $discount = (float) (optional($user->seller_package)->prod_discount ?? 0);

    return view('addon:multivendor::seller.my_orders.place_order', compact('discount'));
  }

  /* ---------- My orders list ---------- */
  public function my_orders(Request $request)
  {
    $user = auth()->user();

    $orders = Order::where('user_id', $user->id)
      ->whereNotNull('code');

    if ($s = $request->get('search')) {
      $orders->where('code', 'like', "%{$s}%");
    }
    if ($ps = $request->get('payment_status')) {
      $orders->where('payment_status', $ps);
    }
    if ($ds = $request->get('delivery_status')) {
      $orders->where('delivery_status', $ds);
    }

    $orders   = $orders->latest()->paginate(15);
    $discount = (float) (optional($user->seller_package)->prod_discount ?? 0);

    return view('addon:multivendor::seller.my_orders.index', [
      'orders'          => $orders,
      'payment_status'  => $request->get('payment_status'),
      'delivery_status' => $request->get('delivery_status'),
      'sort_search'     => $request->get('search'),
      'discount'        => $discount,
    ]);
  }

  /* ---------- Product search for POS ---------- */
  public function search(Request $request)
  {
    $products = Product::query()
      ->where('approved', 1)
      ->where('published', 1);

    if ($kw = $request->get('keyword')) {
      $products->where('name', 'like', "%{$kw}%");
    }

    return new PosProductCollection($products->paginate(16));
  }

  /* ---------- Cart ops ---------- */
  public function addToCart(Request $request)
  {
    $product = Product::findOrFail((int) $request->product_id);

    // store plain arrays (not Eloquent models) to the session
    $data = [
      'id'       => (int) $product->id,
      'quantity' => max(1, (int) $product->min_qty),
      'price'    => (float) $product->lowest_price,
      'name'     => (string) $product->name,
      'thumb_id' => $product->thumbnail_img, // int or null
    ];

    $existing = Session::get('pos.cart', []);
    // normalize to a simple PHP array
    if ($existing instanceof Collection) $existing = $existing->toArray();

    $cart = [];
    $found = false;
    foreach ((array) $existing as $row) {
      $row = $this->toArray($row);
      if ((int) data_get($row, 'id', 0) === (int) $product->id) {
        $found = true;
        $row['quantity'] = (int) data_get($row, 'quantity', 0) + 1;
      }
      $cart[] = $row;
    }
    if (!$found) $cart[] = $data;

    Session::put('pos.cart', $cart);

    $discount = (float) (optional(auth()->user()->seller_package)->prod_discount ?? 0);

    return [
      'success' => 1,
      'message' => '',
      'view'    => view('backend.pos.cart', compact('discount'))->render(),
    ];
  }

  public function updateQuantity(Request $request)
  {
    $existing = Session::get('pos.cart', []);
    if ($existing instanceof Collection) $existing = $existing->toArray();

    $idx = (int) $request->key;
    $qty = max(1, (int) $request->quantity);

    foreach ($existing as $k => &$row) {
      if ($k === $idx) {
        $row = $this->toArray($row);
        $row['quantity'] = $qty;
      }
    }
    unset($row);

    Session::put('pos.cart', $existing);

    $discount = (float) (optional(auth()->user()->seller_package)->prod_discount ?? 0);

    return [
      'success' => 1,
      'message' => '',
      'view'    => view('backend.pos.cart', compact('discount'))->render(),
    ];
  }

  public function removeFromCart(Request $request)
  {
    $existing = Session::get('pos.cart', []);
    if ($existing instanceof Collection) $existing = $existing->toArray();

    $idx = (int) $request->key;
    if (array_key_exists($idx, $existing)) {
      unset($existing[$idx]);
      $existing = array_values($existing);
      Session::put('pos.cart', $existing);
    }

    $discount = (float) (optional(auth()->user()->seller_package)->prod_discount ?? 0);

    return view('backend.pos.cart', compact('discount'));
  }

  /* Optional: allow manual discount override from UI */
  public function setDiscount(Request $request)
  {
    $val = (float) $request->get('discount', 0);
    $val = max(0, min(100, $val));
    Session::put('pos.discount', $val);
    return response()->json(['success' => 1]);
  }

  /* ---------- Order summary (modal content) ---------- */
  public function get_order_summary(Request $request)
  {
    $user = Auth::user();

    // % Discount (prefer session, fallback to user's seller package)
    $discount = (float) Session::get(
      'pos.discount',
      (float) (optional($user->seller_package)->prod_discount ?? 0)
    );

    // ---- Normalize cart from session to a clean array of rows ----
    $raw = Session::get('pos.cart', []);
    if ($raw instanceof \Illuminate\Support\Collection) {
      $raw = $raw->all();
    } elseif (!is_array($raw)) {
      $raw = (array) $raw;
    }

    // Extract valid rows (id & qty > 0). Keep any cart-provided price/name/thumb if present.
    $normalized = [];
    foreach ($raw as $row) {
      $pid = (int) ($row['id'] ?? $row['product_id'] ?? 0);
      $qty = (int) ($row['quantity'] ?? $row['qty'] ?? 0);
      if ($pid <= 0 || $qty <= 0) {
        continue;
      }

      $normalized[] = [
        'id'       => $pid,
        'quantity' => $qty,
        'price'    => array_key_exists('price', $row) ? (float) $row['price'] : null, // may be null -> fallback
        'name'     => $row['name']     ?? null,
        'thumb_id' => $row['thumb_id'] ?? null,
      ];
    }

    // Batch-load products for enrichment & fallback values
    $ids      = array_column($normalized, 'id');
    $products = $ids
      ? \App\Models\Product::whereIn('id', $ids)->get()->keyBy('id')
      : collect();

    $cart     = [];
    $subtotal = 0.0;

    foreach ($normalized as $item) {
      $p = $products->get($item['id']);

      $price = !is_null($item['price'])
        ? (float) $item['price']
        : (float) optional($p)->lowest_price;

      $qty   = (int) $item['quantity'];
      $line  = max(0.0, $price) * max(0, $qty);

      $cart[] = [
        'id'       => $item['id'],
        'name'     => $item['name']     ?? ($p->name ?? translate('Unknown product')),
        'thumb_id' => $item['thumb_id'] ?? ($p->thumbnail_img ?? null),
        'price'    => $price,
        'quantity' => $qty,
        'total'    => $line,
        'product'  => $p, // optional if your blade needs it
      ];

      $subtotal += $line;
    }

    // Final totals
    $total = $subtotal - ($subtotal * $discount / 100);

    // ---- Shipping snapshot (user vs recipient) ----
    $shipToOther   = (bool) ($user->ship_to_other ?? false);
    $shippingName  = $shipToOther ? ($user->recipient_name    ?: '-') : ($user->name    ?: '-');
    $shippingEmail = $user->email ?: '-';
    $shippingPhone = $shipToOther ? ($user->recipient_phone   ?: '-') : ($user->phone   ?: '-');
    $shippingAddr  = $shipToOther ? ($user->recipient_address ?: '-') : ($user->address ?: '-');

    // Persist snapshot for order placement
    Session::put('pos.shipping_info', [
      'ship_to_other'     => $shipToOther,
      'name'              => $shippingName,
      'email'             => $shippingEmail,
      'phone'             => $shippingPhone,
      'address'           => $shippingAddr,
      'recipient_name'    => $user->recipient_name,
      'recipient_phone'   => $user->recipient_phone,
      'recipient_address' => $user->recipient_address,
    ]);
    Session::put('pos.discount', $discount);

    return view('backend.pos.order_summary', [
      'user'            => $user,
      'cart'            => $cart,        // <-- guaranteed array
      'subtotal'        => $subtotal,
      'discount'        => $discount,
      'total'           => $total,
      'shipToOther'     => $shipToOther,
      'shippingName'    => $shippingName,
      'shippingEmail'   => $shippingEmail,
      'shippingPhone'   => $shippingPhone,
      'shippingAddress' => $shippingAddr,
    ]);
  }


  /* ---------- Order create ---------- */
  public function order_store(Request $request)
  {
    try {
      $user = Auth::user();

      // Use the same normalization the summary uses
      $cart = $this->normalizeCartPayload(Session::get('pos.cart', []));

      if (count($cart) === 0) {
        // Don’t clear the cart automatically; let the user retry
        return ['success' => 0, 'message' => translate('Please select a product')];
      }

      // Shipping snapshot
      $shipping_info = (array) Session::get('pos.shipping_info', []);
      if (empty($shipping_info)) {
        $shipToOther = (bool) $user->ship_to_other;
        $shipping_info = [
          'ship_to_other' => $shipToOther,
          'name'          => $shipToOther ? ($user->recipient_name ?? $user->name) : $user->name,
          'email'         => $user->email,
          'phone'         => $shipToOther ? ($user->recipient_phone ?? $user->phone) : $user->phone,
          'address'       => $shipToOther ? ($user->recipient_address ?? '') : ($user->address ?? ''),
        ];
      }

      $shippingAddress = (string) ($shipping_info['address'] ?? '');
      if ($shippingAddress === '') {
        return ['success' => 0, 'message' => translate('Please set your Shipping Address before placing an order.')];
      }

      // Combined Order
      $combined_order                   = new CombinedOrder;
      $combined_order->code             = date('Ymd-His') . rand(10, 99);
      $combined_order->shipping_address = $shippingAddress;
      // keep same for billing if your table has it; if not, ignore
      if (\Schema::hasColumn('combined_orders', 'billing_address')) {
        $combined_order->billing_address  = $shippingAddress;
      }
      $combined_order->grand_total      = 0;
      $combined_order->save();

      // Order (avoid columns you don’t have)
      $order = Order::create([
        'user_id'           => $user->id,
        'combined_order_id' => $combined_order->id,
        'name'              => $user->name,
        'email'             => $user->email,
        'phone'             => $user->phone,
        'code'              => 1,
        'shipping_address'  => $combined_order->shipping_address,
        'payment_type'      => 'cod',
        'payment_status'    => 'unpaid',
      ]);

      // Items + totals
      $subtotal = 0.0;
      foreach ($cart as $row) {
        $lineTotal = (float) $row['price'] * (int) $row['quantity'];
        $subtotal += $lineTotal;

        OrderDetail::create([
          'order_id'   => $order->id,
          'product_id' => (int) $row['id'],
          'price'      => (float) $row['price'],
          'total'      => $lineTotal,
          'quantity'   => (int) $row['quantity'],
        ]);
      }

      $grand_total = $subtotal + (float) Session::get('pos.shipping', 0);
      $discountPct = (float) Session::get('pos.discount', (float) optional($user->seller_package)->prod_discount ?? 0);
      if ($discountPct > 0) {
        $grand_total -= $grand_total * ($discountPct / 100);
      }

      $order->grand_total = $grand_total;
      $order->save();

      $combined_order->grand_total = $grand_total;
      $combined_order->save();

      OrderUpdate::create([
        'order_id' => $order->id,
        'user_id'  => $user->id,
        'note'     => 'Order has been placed.',
      ]);

      // Emails (best-effort)
      $array = [
        'view'    => 'emails.invoice',
        'subject' => 'Your order has been placed - ' . $order->code,
        'from'    => env('MAIL_USERNAME'),
        'order'   => $order,
      ];

      // Notify sellers
      $seller_products = [];
      foreach ($order->orderDetails as $od) {
        if ($od->product && $od->product->user_id) {
          $seller_products[$od->product->user_id][] = $od->product->id;
        }
      }
      foreach ($seller_products as $sellerId => $ids) {
        try {
          if ($email = optional(User::find($sellerId))->email) {
            Mail::to($email)->queue(new \App\Mail\InvoiceEmailManager($array));
          }
        } catch (\Throwable $e) {
        }
      }

      if (env('MAIL_USERNAME')) {
        try {
          Mail::to($user->email)->queue(new \App\Mail\InvoiceEmailManager($array));
          if ($admin = User::where('user_type', 'admin')->first()) {
            Mail::to($admin->email)->queue(new \App\Mail\InvoiceEmailManager($array));
          }
        } catch (\Throwable $e) {
        }
      }

      // Clear POS session
      Session::forget(['pos.shipping_info', 'pos.shipping', 'pos.discount', 'pos.cart']);

      return ['success' => 1, 'message' => translate('Order Completed Successfully.')];
    } catch (\Throwable $e) {
      \Log::error('POS order_store failed: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
      return response()->json([
        'success' => 0,
        'message' => translate('Something went wrong while placing the order. Please try again.'),
      ], 500);
    }
  }

  /* ---------- Save shipping prefs ---------- */
  public function store_shipping(Request $request)
  {
    $validated = $request->validate([
      'ship_to_other'     => ['nullable', 'boolean'],
      'address'           => ['required_unless:ship_to_other,1', 'nullable', 'string', 'max:2000'],
      'recipient_name'    => ['required_if:ship_to_other,1', 'nullable', 'string', 'max:255'],
      'recipient_phone'   => ['required_if:ship_to_other,1', 'nullable', 'string', 'max:32'],
      'recipient_address' => ['required_if:ship_to_other,1', 'nullable', 'string', 'max:2000'],
    ]);

    $user = Auth::user();
    $shipToOther = $request->boolean('ship_to_other');
    $user->ship_to_other = $shipToOther;

    if ($shipToOther) {
      $user->recipient_name    = $validated['recipient_name']    ?? $user->recipient_name;
      $user->recipient_phone   = $validated['recipient_phone']   ?? $user->recipient_phone;
      $user->recipient_address = $validated['recipient_address'] ?? $user->recipient_address;

      Session::put('pos.shipping_info', [
        'ship_to_other'     => true,
        'name'              => $user->recipient_name,
        'email'             => $user->email,
        'phone'             => $user->recipient_phone,
        'address'           => $user->recipient_address,
        'recipient_name'    => $user->recipient_name,
        'recipient_phone'   => $user->recipient_phone,
        'recipient_address' => $user->recipient_address,
      ]);
    } else {
      if (array_key_exists('address', $validated)) {
        $user->address = $validated['address'];
      }

      Session::put('pos.shipping_info', [
        'ship_to_other' => false,
        'name'          => $user->name,
        'email'         => $user->email,
        'phone'         => $user->phone,
        'address'       => $user->address,
      ]);
    }

    $user->save();

    flash(translate('Shipping preferences updated.'))->success();
    return back();
  }

  /* ---------- Helpers ---------- */

  /** Normalize any array/object/Collection cart payload into enriched rows ready for summary/order creation. */
  private function normalizeCartPayload($raw): array
  {
    if ($raw instanceof Collection) $raw = $raw->toArray();
    $rows = [];

    // Accept both array and single row
    foreach ((array) $raw as $row) {
      $row = $this->toArray($row);

      // try all likely key names
      $id  = (int) data_get($row, 'id', data_get($row, 'product_id', 0));
      $qty = (int) data_get($row, 'quantity', data_get($row, 'qty', 0));
      if ($id <= 0 || $qty <= 0) continue;

      $product = Product::find($id);
      if (!$product) continue;

      $price = data_get($row, 'price');
      $price = is_null($price) ? (float) $product->lowest_price : (float) $price;

      $rows[] = [
        'id'       => $id,
        'name'     => data_get($row, 'name', $product->name),
        'thumb_id' => data_get($row, 'thumb_id', $product->thumbnail_img),
        'price'    => $price,
        'quantity' => $qty,
        'total'    => $price * $qty,
        'product'  => $product,
      ];
    }

    return $rows;
  }

  /** Cast mixed (array|object) to a plain PHP array safely. */
  private function toArray($mixed): array
  {
    if (is_array($mixed)) return $mixed;
    if ($mixed instanceof Collection) return $mixed->toArray();
    if (is_object($mixed)) {
      // Best-effort: convert public props to array
      return json_decode(json_encode($mixed), true) ?: [];
    }
    return (array) $mixed;
  }
}
