<?php

namespace App\Addons\Multivendor\Http\Controllers\Seller;

use Auth;
use App\Models\City;
use App\Models\Shop;
use App\Models\User;
use App\Models\State;
use App\Models\Region;
use App\Models\Barangay;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\SellerPackage;
use App\Models\ManualPaymentMethod;
use App\Http\Controllers\Controller;
use App\Models\SellerPackagePayment;
use Illuminate\Support\Facades\Validator;

class SellerPackageController extends Controller
{

  public function select_package()
  {
    $user = auth()->user();

    $seller_packages = SellerPackage::all();

    // Works whether you store the FK or rely on the relation
    $user_seller_package_id = $user->seller_package_id ?? optional($user->sellerPackage)->id;

    $seller_package_payments = SellerPackagePayment::where('user_id', $user->id)
      ->pluck('seller_package_id')
      ->toArray();

    $disabledPackages = [];
    foreach ($seller_packages as $seller_package) {
      if (in_array($seller_package->id, $seller_package_payments, true)) {
        $disabledPackages[] = $seller_package->id;
      }
    }

    return view(
      'addon:multivendor::seller.package.package_select',
      compact('seller_packages', 'user_seller_package_id', 'disabledPackages')
    );
  }

  public function package_purchase_modal(Request $request)
  {
    $seller_packages = SellerPackage::findOrFail($request->id);

    // default empty collections
    $cities = collect();
    $states = collect();
    $barangays = collect();
    $regions = collect();

    switch ((int) $seller_packages->geographic_level) {
      case 1: // barangay-level => pick a CITY first, barangays come from API
        $cities = City::where('status', 1)->orderBy('name')->get();
        break;

      case 2: // city-level
        $cities = City::whereNull('seller_assign')->where('status', 1)->orderBy('name')->get();
        break;

      case 3: // province/state-level
        $states = State::whereNull('seller_assign')->where('status', 1)->orderBy('name')->get();
        break;

      case 4: // region-level
        $regions = Region::whereNull('seller_assign')->where('status', 1)->orderBy('name')->get();
        break;

        // case 0 or anything else => no location selection needed
    }

    // IMPORTANT: include 'regions' in compact (it was missing)
    return view(
      'addon:multivendor::seller.package.package_purchase_modal',
      compact('seller_packages', 'cities', 'states', 'barangays', 'regions')
    );
  }

  // public function package_purchase_store(Request $request)
  // {
  //   $userId    = auth()->id();
  //   $packageId = (int) $request->id;

  //   // Ensure package exists
  //   $sellerPackage = \App\Models\SellerPackage::findOrFail($packageId);
  //   $level = (int) $sellerPackage->geographic_level;

  //   // Base rules
  //   $rules = [
  //     'id' => ['required', 'exists:seller_packages,id'],
  //   ];

  //   // Validation based on geographic_level (0..4)
  //   switch ($level) {
  //     case 1: // city + barangay (assign barangay)
  //       $rules['selected_city'] = ['required', 'exists:cities,id'];
  //       $rules['selected_barangay'] = [
  //         'required',
  //         'exists:barangays,id',
  //         function ($attribute, $value, $fail) use ($request) {
  //           $barangay = \App\Models\Barangay::find($value);
  //           if ($barangay && $request->filled('selected_city') && (int) $barangay->city_id !== (int) $request->selected_city) {
  //             $fail(__('Selected barangay does not belong to the chosen city.'));
  //           }
  //         }
  //       ];
  //       break;

  //     case 2: // city only
  //       $rules['selected_city'] = ['required', 'exists:cities,id'];
  //       break;

  //     case 3: // state/province
  //       $rules['selected_state'] = ['required', 'exists:states,id'];
  //       break;

  //     case 4: // region
  //       $rules['selected_region'] = ['required', 'exists:regions,id'];
  //       break;

  //     case 0:
  //     default:
  //       // no location required
  //       break;
  //   }

  //   $validator = Validator::make($request->all(), $rules);
  //   if ($validator->fails()) {
  //     $firstError = $validator->errors()->first();
  //     if ($firstError) {
  //       flash($firstError)->error();
  //     }
  //     return back()->withErrors($validator)->withInput()->with('package_modal_id', $sellerPackage->id);
  //   }

  //   // Derive the location ID the user is purchasing (by level)
  //   $locationId = null;
  //   switch ($level) {
  //     case 1:
  //       $locationId = (int) $request->selected_barangay;
  //       break;
  //     case 2:
  //       $locationId = (int) $request->selected_city;
  //       break;
  //     case 3:
  //       $locationId = (int) $request->selected_state;
  //       break;
  //     case 4:
  //       $locationId = (int) $request->selected_region;
  //       break;
  //     case 0:
  //     default:
  //       $locationId = null;
  //       break;
  //   }

  //   // If a location is required but we somehow failed to resolve it
  //   if ($level > 0 && empty($locationId)) {
  //     flash(__('Please select a valid location.'))->error();
  //     return back()->withInput()->with('package_modal_id', $sellerPackage->id);
  //   }

  //   /**
  //    * HARD CHECK: Has this location ALREADY been assigned to a seller?
  //    * We check geo tables' `seller_assign` column:
  //    */
  //   if ($level > 0) {
  //     $alreadyAssigned = false;

  //     switch ($level) {
  //       case 1: // barangay must also match the chosen city
  //         $b = \App\Models\Barangay::findOrFail($locationId);
  //         if ($request->filled('selected_city') && (int) $b->city_id !== (int) $request->selected_city) {
  //           flash(__('Selected barangay does not belong to the chosen city.'))->error();
  //           return back()->withInput()->with('package_modal_id', $sellerPackage->id);
  //         }
  //         $alreadyAssigned = !is_null($b->seller_assign);
  //         break;

  //       case 2:
  //         $c = \App\Models\City::findOrFail($locationId);
  //         $alreadyAssigned = !is_null($c->seller_assign);
  //         break;

  //       case 3:
  //         $s = \App\Models\State::findOrFail($locationId);
  //         $alreadyAssigned = !is_null($s->seller_assign);
  //         break;

  //       case 4:
  //         $r = \App\Models\Region::findOrFail($locationId);
  //         $alreadyAssigned = !is_null($r->seller_assign);
  //         break;
  //     }

  //     if ($alreadyAssigned) {
  //       // Hard stop: truly unavailable
  //       flash(__('Selected location is not available. Please choose another.'))->error();
  //       return back()->withInput()->with('package_modal_id', $sellerPackage->id);
  //     }
  //   }

  //   /**
  //    * SOFT CHECK: Is someone else already intending to buy the same (level + location)?
  //    * If yes, we still allow creation, but we’ll show a warning after commit.
  //    */
  //   $someoneIntends = false;
  //   if ($level > 0) {
  //     $someoneIntends = \App\Models\SellerPackagePayment::where('geographic_level', $level)
  //       ->where('seller_assign_location', $locationId)
  //       ->where('user_id', '<>', $userId)
  //       ->whereIn('approval', [2, 3]) // 2=pending, 3=proof uploaded
  //       ->exists();
  //   }

  //   try {
  //     DB::beginTransaction();

  //     $purchase                          = new \App\Models\SellerPackagePayment();
  //     $purchase->user_id                 = $userId;
  //     $purchase->seller_package_id       = $sellerPackage->id;
  //     $purchase->amount                  = $sellerPackage->amount;
  //     $purchase->approval                = 2; // pending/under review
  //     $purchase->geographic_level        = $level;
  //     $purchase->seller_assign_location  = $locationId; // store the numeric ID
  //     $purchase->save(); // get ID

  //     // Generate a stable transaction id now that we have an ID
  //     $purchase->transaction_id = sprintf(
  //       '%d-%d-%d-%s',
  //       $purchase->id,
  //       $userId,
  //       $sellerPackage->id,
  //       Str::upper(Str::random(6))
  //     );

  //     $purchase->geographic_level = (int) $sellerPackage->geographic_level;

  //     $purchase->save();

  //     DB::commit();

  //     if ($someoneIntends) {
  //       flash(__('There is already someone intending to buy this location. Please pay as soon as possible to get this location.'))->warning();
  //     }

  //     flash(translate('Purchase created. Please proceed to payment.'))->success();
  //     return back();
  //   } catch (\Throwable $e) {
  //     DB::rollBack();
  //     \Log::error('Package purchase failed', ['error' => $e->getMessage()]);
  //     flash(__('Unable to complete purchase. Please try again.'))->error();
  //     return back()->withInput()->with('package_modal_id', $sellerPackage->id);
  //   }
  // }

  public function package_purchase_store(Request $request)
  {
    $userId    = auth()->id();
    $packageId = (int) $request->id;

    // Ensure package exists
    $sellerPackage = \App\Models\SellerPackage::findOrFail($packageId);
    $level = (int) $sellerPackage->geographic_level;

    // Base rules
    $rules = [
      'id' => ['required', 'exists:seller_packages,id'],
    ];

    // Validation based on geographic_level (0..4)
    switch ($level) {
      case 1: // city + barangay (assign barangay)
        $rules['selected_city'] = ['required', 'exists:cities,id'];
        $rules['selected_barangay'] = [
          'required',
          'exists:barangays,id',
          function ($attribute, $value, $fail) use ($request) {
            $barangay = \App\Models\Barangay::find($value);
            if ($barangay && $request->filled('selected_city') && (int) $barangay->city_id !== (int) $request->selected_city) {
              $fail(__('Selected barangay does not belong to the chosen city.'));
            }
          }
        ];
        break;

      case 2: // city only
        $rules['selected_city'] = ['required', 'exists:cities,id'];
        break;

      case 3: // state/province
        $rules['selected_state'] = ['required', 'exists:states,id'];
        break;

      case 4: // region
        $rules['selected_region'] = ['required', 'exists:regions,id'];
        break;

      case 0:
      default:
        // no location required
        break;
    }

    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      $firstError = $validator->errors()->first();
      if ($firstError) {
        flash($firstError)->error();
      }
      return back()->withErrors($validator)->withInput()->with('package_modal_id', $sellerPackage->id);
    }

    // Derive the location ID the user is purchasing (by level)
    $locationId = null;
    switch ($level) {
      case 1:
        $locationId = (int) $request->selected_barangay;
        break;
      case 2:
        $locationId = (int) $request->selected_city;
        break;
      case 3:
        $locationId = (int) $request->selected_state;
        break;
      case 4:
        $locationId = (int) $request->selected_region;
        break;
      case 0:
      default:
        $locationId = null;
        break;
    }

    if ($level > 0 && empty($locationId)) {
      flash(__('Please select a valid location.'))->error();
      return back()->withInput()->with('package_modal_id', $sellerPackage->id);
    }

    // HARD CHECK: location already assigned?
    if ($level > 0) {
      $alreadyAssigned = false;
      switch ($level) {
        case 1:
          $b = \App\Models\Barangay::findOrFail($locationId);
          if ($request->filled('selected_city') && (int) $b->city_id !== (int) $request->selected_city) {
            flash(__('Selected barangay does not belong to the chosen city.'))->error();
            return back()->withInput()->with('package_modal_id', $sellerPackage->id);
          }
          $alreadyAssigned = !is_null($b->seller_assign);
          break;
        case 2:
          $c = \App\Models\City::findOrFail($locationId);
          $alreadyAssigned = !is_null($c->seller_assign);
          break;
        case 3:
          $s = \App\Models\State::findOrFail($locationId);
          $alreadyAssigned = !is_null($s->seller_assign);
          break;
        case 4:
          $r = \App\Models\Region::findOrFail($locationId);
          $alreadyAssigned = !is_null($r->seller_assign);
          break;
      }
      if ($alreadyAssigned) {
        flash(__('Selected location is not available. Please choose another.'))->error();
        return back()->withInput()->with('package_modal_id', $sellerPackage->id);
      }
    }

    // SOFT CHECK: someone else intends to buy same location?
    $someoneIntends = false;
    if ($level > 0) {
      $someoneIntends = \App\Models\SellerPackagePayment::where('geographic_level', $level)
        ->where('seller_assign_location', $locationId)
        ->where('user_id', '<>', $userId)
        ->whereIn('approval', [2, 3]) // 2=pending, 3=proof uploaded
        ->exists();
    }

    try {
      DB::beginTransaction();

      $purchase                          = new \App\Models\SellerPackagePayment();
      $purchase->user_id                 = $userId;
      $purchase->seller_package_id       = $sellerPackage->id;
      $purchase->amount                  = $sellerPackage->amount;
      $purchase->approval                = 2; // pending/under review
      $purchase->geographic_level        = $level;
      $purchase->seller_assign_location  = $locationId; // store numeric ID
      $purchase->save();

      // Generate a transaction id
      $purchase->transaction_id = sprintf(
        '%d-%d-%d-%s',
        $purchase->id,
        $userId,
        $sellerPackage->id,
        Str::upper(Str::random(6))
      );
      $purchase->geographic_level = (int) $sellerPackage->geographic_level;
      $purchase->save();

      DB::commit();

      // Optional warning if contested location
      if ($someoneIntends) {
        flash(__('There is already someone intending to buy this location. Please pay as soon as possible to get this location.'))->warning();
      }

      // ✅ Your requested success flash + redirect
      flash(translate('Purchase has been created. Please proceed with uploading your payment.'))->success();
      return redirect()->route('seller.package_purchase_history');
    } catch (\Throwable $e) {
      DB::rollBack();
      \Log::error('Package purchase failed', ['error' => $e->getMessage()]);
      flash(__('Unable to complete purchase. Please try again.'))->error();
      return back()->withInput()->with('package_modal_id', $sellerPackage->id);
    }
  }

  public function package_payment_modal(Request $request)
  {
    $package_payment = SellerPackagePayment::findOrFail($request->id);
    $seller_package_id = $package_payment->seller_package_id;
    $amount = $package_payment->amount;
    $transaction_id = $package_payment->transaction_id;
    $manual_payment_methods = ManualPaymentMethod::all();
    return view('addon:multivendor::seller.package.package_payment_modal', compact('package_payment', 'seller_package_id', 'amount', 'transaction_id', 'manual_payment_methods'));
  }

  public function package_payment_proof(Request $request)
  {
    $transaction_id = $request->transaction_id;
    $userId = $request->userId;
    $seller_package_id = $request->seller_package_id;
    $payment_method = $request->payment_method;

    // Find the seller package payment record with the given transaction_id
    $payment = SellerPackagePayment::where('transaction_id', $transaction_id)->first();

    if ($payment) {
      // Update the payment proof fields
      $payment->payment_proof_upload = $request->payment_proof_upload;
      $payment->approval = 3;
      $payment->self_purchased = 1;
      $payment->payment_proof_tran_id = $payment->id . Str::random(5) . $userId . $seller_package_id;
      $payment->payment_method = $payment_method;
      $payment->save();



      flash(translate('Payment proof uploaded successfully'))->success();
      return back();
    } else {
      flash(translate('Seller package payment not found for transaction ID: ' . $transaction_id))->error();
      return back();
    }
  }

  public function package_purchase(Request $request)
  {

    $seller_package = SellerPackage::findOrFail($request->seller_package_id);

    // if ($seller_package->product_upload_limit < Auth::user()->shop->products->count()){
    //     flash(translate('You have more uploaded products than this package limit. You need to remove excessive products to downgrade.'))->warning();
    //     return back();
    // }

    if ($seller_package->amount == 0) {
      return $this->purchase_payment_done($seller_package->id, null, null);
    } else {

      $request->redirect_to = null;
      $request->amount = $seller_package->amount;
      $request->payment_method = $request->payment_option;
      $request->payment_type = 'seller_package_payment';
      $request->user_id = auth()->user()->id;
      $request->order_code = null;
      $request->seller_package_id = $request->seller_package_id;

      //return (new PaymentController())->payment_initialize($request, $request->payment_option);
    }
  }

  public function purchase_payment_done($package_id, $payment_method, $payment_data)
  {
    $shop = Auth::user()->shop;
    $seller_package = SellerPackage::findOrFail($package_id);
    $shop->seller_package_id = $seller_package->id;
    $shop->product_upload_limit = $seller_package->product_upload_limit;
    $shop->commission = $seller_package->commission;
    $shop->published = 1;
    $shop->package_invalid_at = date('Y-m-d', strtotime($seller_package->duration . 'days'));
    $shop->save();

    if ($payment_method != null) {
      $seller_package_payment = new SellerPackagePayment;
      $seller_package_payment->user_id = Auth::user()->id;
      $seller_package_payment->seller_package_id = $package_id;
      $seller_package_payment->amount = $seller_package->amount;
      $seller_package_payment->payment_method = $payment_method;
      $seller_package_payment->payment_details = $payment_data;
      $seller_package_payment->save();
    }

    flash(translate('Package purchasing successful'))->success();
    return redirect()->route('seller.dashboard');
  }

  public function package_purchase_history()
  {
    $package_payments = SellerPackagePayment::where('user_id', Auth::user()->id)->latest()->paginate(20);
    return view('addon:multivendor::seller.package.payment_history', compact('package_payments'));
  }
}
