<?php

namespace App\Addons\Multivendor\Http\Controllers\Seller;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SellerPayout;
use Carbon\Carbon;

class SellerPayoutController extends Controller
{
  /**
   * Display a listing of the resource.
   *
   * @return \Illuminate\Http\Response
   */
  public function index(Request $request)
  {
    $user = auth()->user();
    $date_range = $request->date_range;

    $payouts = SellerPayout::with('user')              // was ->with('shop')
      ->where('status', 'paid')
      ->where('user_id', $user->id);                 // was ->where('shop_id', $shop_id)

    if (!empty($date_range)) {
      // Expecting "YYYY-MM-DD / YYYY-MM-DD"
      [$from, $to] = array_map('trim', explode(' / ', $date_range));
      $from = Carbon::parse($from)->startOfDay();
      $to   = Carbon::parse($to)->endOfDay();

      $payouts->whereBetween('created_at', [$from, $to]);
    }

    $payouts = $payouts->latest()->paginate(20);

    return view('addon:multivendor::seller.earnings.payouts', compact('payouts', 'date_range'));
  }

  public function payout_requests()
  {
    $user = auth()->user();

    // Payout methods enabled + have the required info?
    $gcashEnabled = (int) ($user->gcash_payout_status ?? 0) === 1
      && filled($user->gcash_name) && filled($user->gcash_no);

    $bankEnabled  = (int) ($user->bank_payout_status ?? 0) === 1
      && filled($user->bank_name) && filled($user->bank_acc_name) && filled($user->bank_acc_no);

    $canRequest = $gcashEnabled || $bankEnabled;

    // Keep showing only pending (“requested”) like your current method
    $payout_requests = \App\Models\SellerPayout::where('user_id', $user->id)
      ->where('status', 'requested')
      ->latest()
      ->paginate(15);

    $currentBalance = (float) ($user->current_balance ?? 0);

    return view('addon:multivendor::seller.earnings.payout_requests', compact(
      'payout_requests',
      'currentBalance',
      'gcashEnabled',
      'bankEnabled',
      'canRequest',
      'user'
    ));
  }


  // public function store_withdraw_request(Request $request)
  // {
  //   $request->validate([
  //     'amount'  => ['required', 'numeric', 'min:1'],
  //     'message' => ['nullable', 'string', 'max:2000'],
  //   ]);

  //   $user = auth()->user();

  //   // Optional: enforce a site-wide minimum withdrawal amount
  //   $min = (float) get_setting('min_withdrawal_amount', 0);
  //   if ($min > 0 && $request->amount < $min) {
  //     flash(translate('Minimum withdrawal amount is :amount', ['amount' => format_price($min)]))->error();
  //     return back();
  //   }

  //   // Check available balance on the USERS table
  //   $available = (float) ($user->current_balance ?? 0);
  //   if ($available < (float) $request->amount) {
  //     flash(translate('You can not request more than your balance'))->error();
  //     return back();
  //   }

  //   // Create payout request against USER, not SHOP
  //   $seller_withdraw_request              = new \App\Models\SellerPayout();
  //   $seller_withdraw_request->user_id     = $user->id;           // ← switched from shop_id
  //   $seller_withdraw_request->requested_amount = (float) $request->amount;
  //   $seller_withdraw_request->seller_note = $request->message;
  //   $seller_withdraw_request->status      = 'requested';

  //   if ($seller_withdraw_request->save()) {
  //     flash(translate('Request has been sent successfully'))->success();
  //     return redirect()->route('seller.payouts.request');
  //   }

  //   flash(translate('Something went wrong'))->error();
  //   return back();
  // }


  public function store_withdraw_request(Request $request)
  {
    $request->validate([
      'amount'         => ['required', 'numeric', 'min:1'],
      'message'        => ['nullable', 'string', 'max:2000'],
      'payment_method' => ['required', 'in:gcash,bank'],
    ]);

    $user = auth()->user();

    // check method is enabled + complete
    if ($request->payment_method === 'gcash') {
      $gcashEnabled = (int) ($user->gcash_payout_status ?? 0) === 1
        && filled($user->gcash_name) && filled($user->gcash_no);
      if (!$gcashEnabled) {
        flash(translate('GCash payout is not configured.'))->error();
        return back();
      }
    } else {
      $bankEnabled = (int) ($user->bank_payout_status ?? 0) === 1
        && filled($user->bank_name) && filled($user->bank_acc_name) && filled($user->bank_acc_no);
      if (!$bankEnabled) {
        flash(translate('Bank payout is not configured.'))->error();
        return back();
      }
    }

    // min withdrawal (optional)
    $min = (float) get_setting('min_withdrawal_amount', 0);
    if ($min > 0 && (float)$request->amount < $min) {
      flash(translate('Minimum withdrawal amount is :amount', ['amount' => format_price($min)]))->error();
      return back();
    }

    // balance check
    $available = (float) ($user->current_balance ?? 0);
    if ($available < (float) $request->amount) {
      flash(translate('You can not request more than your balance'))->error();
      return back();
    }

    // snapshot payment details for the request
    $details = [];
    if ($request->payment_method === 'gcash') {
      $details = [
        'gcash_name' => $user->gcash_name,
        'gcash_no'   => $user->gcash_no,
      ];
    } else {
      $details = [
        'bank_name'     => $user->bank_name,
        'bank_acc_name' => $user->bank_acc_name,
        'bank_acc_no'   => $user->bank_acc_no,
      ];
    }

    $payout = new \App\Models\SellerPayout();
    $payout->user_id          = $user->id;
    $payout->requested_amount = (float) $request->amount;
    $payout->seller_note      = $request->message;
    $payout->status           = 'requested';
    $payout->payment_method   = $request->payment_method; // 'gcash' or 'bank'
    $payout->payment_details  = json_encode($details);

    if ($payout->save()) {
      flash(translate('Request has been sent successfully'))->success();
      return redirect()->route('seller.payouts.request');
    }

    flash(translate('Something went wrong'))->error();
    return back();
  }



  public function payout_settings()
  {
    $user = auth()->user();
    return view('addon:multivendor::seller.earnings.money_withdraw_settings', compact('user'));
  }

  public function payout_settings_update(Request $request)
  {
    $user = auth()->user();

    // checkboxes come only when checked
    $user->gcash_payout_status = $request->has('gcash_payout_status') ? 1 : 0;
    $user->bank_payout_status  = $request->has('bank_payout_status')  ? 1 : 0;

    // GCash
    $user->gcash_name = $request->gcash_name;
    $user->gcash_no   = $request->gcash_no;

    // Bank
    $user->bank_name       = $request->bank_name;
    $user->bank_acc_name   = $request->bank_acc_name;
    $user->bank_acc_no     = $request->bank_acc_no;
    $user->save();

    flash(translate('Withdraw settings updated successfully'))->success();
    return back();
  }
}
