<?php

namespace App\Http\Controllers\Api;

use App\Http\Resources\DownloadedProductResource;
use App\Http\Resources\OrderCollection;
use App\Http\Resources\OrderSingleCollection;
use App\Services\Phone;
use App\Models\Address;
use App\Models\City;
use App\Models\CombinedOrder;
use App\Models\Language;
use App\Models\State;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\OrderUpdate;
use App\Models\Product;
use App\Models\Shop;
use App\Models\Upload;
use App\Models\User;
use App\Notifications\DB\OrderDbNotification;
use App\Notifications\OrderPlacedNotification;
use App\Notifications\SellerInvoiceNotification;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use PDF;

class OrderController extends Controller
{
  public function new_post_order(Request $request)
  {
    if ($request->bearerToken() !== 'aqSm*krXb+!~8SI5O_6V73KaYXB5~F') {
      return response()->json(['error' => 'Unauthorized'], 401);
    }

    try {
      $customer       = $request->input('customer');
      $shipping       = $request->input('shipping_address');
      $shipping_cost  = $request->input('shipping_cost');
      $items          = $request->input('items');
      $seller_id      = $request->input('seller_id');
      $payment_method = $request->input('payment_method');

      // PHONE: normalize and validate PH mobile to 11-digit 09XXXXXXXXX
      $normalizedPhone = Phone::normalizePhMobile($customer['phone'] ?? null);
      if (!$normalizedPhone) {
        return response()->json(['error' => 'Invalid Philippine mobile number. Expect 11 digits starting with 09.'], 422);
      }

      $province_id = $shipping['province'] ?? null;
      $stateRec = \App\Models\State::where('id', $province_id)
        ->orWhere('state_code', $province_id)
        ->first();

      $province_name = $stateRec ? $stateRec->name : (string) $province_id;
      $region_id     = $stateRec ? $stateRec->region_id : null;

      $city_name = $shipping['city'] ?? null;
      $cityRec = null;
      if ($city_name && $stateRec) {
        $cityRec = \App\Models\City::where('name', $city_name)
          ->where('state_id', $stateRec->id)
          ->first();
      } elseif ($city_name) {
        $cityRec = \App\Models\City::where('name', $city_name)->first();
      }
      $city_id   = $cityRec ? $cityRec->id : null;
      $city_name = $cityRec ? $cityRec->name : (string) $city_name;

      $barangay_name = $shipping['barangay'] ?? null;
      $barangayRec = null;
      if ($barangay_name && $city_id) {
        $barangayRec = \App\Models\Barangay::where('name', $barangay_name)
          ->where('city_id', $city_id)
          ->first();
      } elseif ($barangay_name) {
        $barangayRec = \App\Models\Barangay::where('name', $barangay_name)->first();
      }
      $barangay_id   = $barangayRec ? $barangayRec->id : null;
      $barangay_name = $barangayRec ? $barangayRec->name : (string) $barangay_name;

      $address = ($shipping['street'] ?? '') . ', ' .
        $barangay_name . ', ' .
        $city_name . ', ' .
        $province_name;

      $combined_order = new CombinedOrder();
      $combined_order->code             = now()->format('Ymd-His') . rand(10, 99);
      $combined_order->shipping_address = $address;
      $combined_order->billing_address  = $address;

      $grand_total = 0;
      foreach ($items as $item) {
        $product = Product::find($item['product_id']);
        if (!$product) continue;
        $grand_total += $product->lowest_price * $item['quantity'];
      }

      $combined_order->grand_total = $grand_total;
      $combined_order->save();

      $order = Order::create([
        'user_id'           => $seller_id,
        'combined_order_id' => $combined_order->id,
        'name'              => $customer['first_name'] . ' ' . $customer['last_name'],
        'email'             => $customer['email'],
        'phone'             => $normalizedPhone, // PHONE: normalized
        'shipping_address'  => $address,
        'barangay_id'          => $barangay_id,
        'city_id'              => $city_id,
        'province_id'          => $stateRec ? $stateRec->id : null,
        'region_id'            => $region_id,
        'grand_total'       => $grand_total,
        'shipping_cost'     => $shipping_cost,
        'delivery_type'     => 'Standard',
        'payment_type'      => $payment_method,
      ]);

      $seller_commission = optional(User::find($seller_id))->commission ?? 0;
      $order->seller_earning = $seller_commission > 0
        ? ($seller_commission * $grand_total) / 100
        : $grand_total;
      $order->save();

      foreach ($items as $item) {
        $product = Product::find($item['product_id']);
        if (!$product) continue;

        OrderDetail::create([
          'order_id'   => $order->id,
          'product_id' => $product->id,
          'price'      => $product->lowest_price,
          'quantity'   => $item['quantity'],
          'total'      => $product->lowest_price * $item['quantity'],
        ]);
      }

      try {

        if (!empty($customer['email'])) {
          Notification::route('mail', $customer['email'])
            ->notify(new OrderPlacedNotification($combined_order));
        } else {
        }
        foreach ($combined_order->orders as $orderRow) {
          $seller = optional($orderRow->orderDetails->first())->product->shop->user ?? null;

          if ($seller && !empty($seller->email)) {
            Notification::route('mail', $seller->email)->notify(new SellerInvoiceNotification($orderRow));
            Notification::send($seller, new SellerInvoiceNotification($orderRow));
            Notification::send($seller, new OrderDbNotification($orderRow));
          } else {
          }
        }
      } catch (\Exception $e) {
      }
      return response()->json(['success' => true]);
    } catch (\Exception $e) {
      return response()->json(['error' => 'Something went wrong.'], 500);
    }
  }

  public function post_store_order(Request $request)
  {
    // Create a combined order
    $combined_order = new CombinedOrder;
    $combined_order->code = date('Ymd-His') . rand(10, 99);

    // Get state details from the request
    $stateCode = $request->billing['state'];
    $state = State::where('state_code', $stateCode)->first();

    // Construct the shipping and billing address
    $shippingAddress = $request->billing['address_1'] . ', ' . $request->billing['address_2'] . ', ' . $request->billing['city'] . ', ' . $state->name;
    $combined_order->shipping_address = $shippingAddress;
    $combined_order->billing_address = $shippingAddress;

    $grand_total = 0;
    $package_number = 1;

    // Process each line item and calculate total
    foreach ($request->line_items as $line_item) {
      $product = Product::find($line_item['product_id']);
      $line_item_total = $product->lowest_price * $line_item['quantity'];
      $grand_total += $line_item_total;
    }

    $delivery_type = $request->shipping_lines[0]['method_title'];
    $shipping_cost = floatval($request->shipping_total);
    $grand_total += $shipping_cost;

    $combined_order->grand_total = $grand_total;
    $combined_order->save();

    // Extract the seller ID from meta_data
    $billing_seller_meta = collect($request->meta_data)->firstWhere('key', '_billing_seller_id');
    $billing_seller_id = $billing_seller_meta['value'];

    // Create the order
    $order = Order::create([
      'shop_id' => $billing_seller_id,
      'combined_order_id' => $combined_order->id,
      'name' => $request->billing['first_name'] . ' ' . $request->billing['last_name'],
      'email' => $request->billing['email'],
      'phone' => $request->billing['phone'],
      'shipping_address' => $combined_order->shipping_address,
      'billing_address' => $combined_order->shipping_address,
      'customer_city' => $request->billing['city'],
      'customer_state' => $state->name,
      'grand_total' => $grand_total,
      'shipping_cost' => $shipping_cost,
      'delivery_type' => $delivery_type,
      'payment_type' => $request->payment_method_title,
    ]);

    // Calculate seller earnings
    $shop_commission = Shop::find($billing_seller_id)->commission;
    $order_price = $order->grand_total - $order->shipping_cost;
    $seller_earning = $shop_commission > 0 ? ($shop_commission * $order_price) / 100 : $order_price;
    $order->seller_earning = $seller_earning;
    $order->save();

    // Create OrderDetail records
    foreach ($request->line_items as $line_item) {
      $product = Product::find($line_item['product_id']);
      $orderDetail = OrderDetail::create([
        'order_id' => $order->id,
        'product_id' => $product->id,
        'price' => $product->lowest_price,
        'quantity' => $line_item['quantity'],
        'total' => $product->lowest_price * $line_item['quantity'],
      ]);
    }

    // Send notifications
    try {
      Notification::route('mail', $request->billing['email'])->notify(new OrderPlacedNotification($combined_order));

      foreach ($combined_order->orders as $order) {
        $seller = $order->orderDetails->first()->product->shop->user;

        // Send Seller Invoice Notification to the shop owner
        Notification::route('mail', $order->shop->user->email)->notify(new SellerInvoiceNotification($order));
        Notification::send($seller, new SellerInvoiceNotification($order));
        Notification::send($seller, new OrderDbNotification($order));
      }
    } catch (\Exception $e) {
      \Log::error('Notification Error: ' . $e->getMessage());
    }

    return response()->json([
      'success' => true,
    ]);
  }

  public function index()
  {
    return new OrderCollection(CombinedOrder::with(['user', 'orders.orderDetails.variation.product', 'orders.orderDetails.variation.combinations', 'orders.shop'])->where('user_id', auth('api')->user()->id)->latest()->paginate(12));
  }

  public function show($order_code)
  {
    $order = CombinedOrder::where('code', $order_code)->with(['user', 'orders.orderDetails.variation.product', 'orders.orderDetails.variation.combinations', 'orders.shop'])->first();
    if ($order) {
      if (auth('api')->user()->id == $order->user_id) {
        return new OrderSingleCollection($order);
      } else {
        return response()->json([
          'success' => false,
          'message' => translate("This order is not your. You can't check details of this order"),
          'status' => 200
        ]);
      }
    } else {
      return response()->json([
        'success' => false,
        'message' => translate("No order found by this code"),
        'status' => 404
      ]);
    }
  }

  public function get_shipping_cost(Request $request, $address_id)
  {
    $address = Address::find($address_id);
    $city = City::find($address->city_id);

    if ($city && $city->zone != null) {
      return response()->json([
        'success' => true,
        'standard_delivery_cost' => $city->zone->standard_delivery_cost,
        'express_delivery_cost' => $city->zone->express_delivery_cost,
      ]);
    } else {
      return response()->json([
        'success' => false,
        'standard_delivery_cost' => 0,
        'express_delivery_cost' => 0,
      ]);
    }
  }
  // public function invoice_download(Request $request, $order_code)
  // {
  //     $currency_code = env('DEFAULT_CURRENCY_CODE');

  //     $language_code = app()->getLocale();

  //     if (optional(Language::where('code', $language_code)->first())->rtl == 1) {
  //         $direction = 'rtl';
  //         $default_text_align = 'right';
  //         $reverse_text_align = 'left';
  //     } else {
  //         $direction = 'ltr';
  //         $default_text_align = 'left';
  //         $reverse_text_align = 'right';
  //     }


  //     if ($currency_code == 'BDT' || $language_code == 'bd') {
  //         // bengali font
  //         $font_family = "'Hind Siliguri','sans-serif'";
  //     } elseif ($currency_code == 'KHR' || $language_code == 'kh') {
  //         // khmer font
  //         $font_family = "'Hanuman','sans-serif'";
  //     } elseif ($currency_code == 'AMD') {
  //         // Armenia font
  //         $font_family = "'arnamu','sans-serif'";
  //     } elseif ($currency_code == 'ILS') {
  //         // Israeli font
  //         $font_family = "'Varela Round','sans-serif'";
  //     } elseif ($currency_code == 'AED' || $currency_code == 'EGP' || $language_code == 'sa' || $currency_code == 'IQD' || $language_code == 'ir') {
  //         // middle east/arabic font
  //         $font_family = "'XBRiyaz','sans-serif'";
  //     } else {
  //         // general for all
  //         $font_family = "'Roboto','sans-serif'";
  //     }

  //     $order = Order::where('code', $order_code)->first();
  //     $pdf = PDF::loadView('backend.invoices.invoice', [
  //         'order' => $order,
  //         'font_family' => $font_family,
  //         'direction' => $direction,
  //         'default_text_align' => $default_text_align,
  //         'reverse_text_align' => $reverse_text_align
  //     ], [], [])->save(public_path('invoices/') . 'order-invoice-' . $order->code . '.pdf');

  //     $pdf = static_asset('invoices/' . 'order-invoice-' . $order->code . '.pdf');

  //     try {
  //         return response()->json([
  //             'success' => true,
  //             'message' => translate('Invoice generated.'),
  //             'invoice_url' => $pdf
  //         ]);
  //     } catch (\Exception $e) {
  //         return response()->json([
  //             'success' => false,
  //             'message' => translate('Something went wrong!'),
  //             'invoice_url' => ''
  //         ]);
  //     }
  // }

  public function cancel($order_id)
  {
    $order = Order::findOrFail($order_id);
    if (auth('api')->user()->id !== $order->user_id) {
      return response()->json(null, 401);
    }

    if ($order->delivery_status == 'order_placed' && $order->payment_status == 'unpaid') {
      $order->delivery_status = 'cancelled';
      $order->save();

      foreach ($order->orderDetails as $orderDetail) {
        try {
          foreach ($orderDetail->product->categories as $category) {
            $category->sales_amount -= $orderDetail->total;
            $category->save();
          }

          $brand = $orderDetail->product->brand;
          if ($brand) {
            $brand->sales_amount -= $orderDetail->total;
            $brand->save();
          }
        } catch (\Exception $e) {
        }
      }

      return response()->json([
        'success' => true,
        'message' => translate("Order has been cancelled"),
      ]);
    } else {
      return response()->json([
        'success' => false,
        'message' => translate("This order can't be cancelled."),
      ]);
    }
  }

  public function store(Request $request)
  {

    $combined_order = new CombinedOrder;
    $combined_order->code = date('Ymd-His') . rand(10, 99);

    $stateCode = $request->billing['state'];
    $state = State::where('state_code', $stateCode)->first();

    $shippingAddress = $request->billing['address_1'] . ', ' . $request->billing['address_2'] . ', ' . $request->billing['city'] . ', ' . $state->name;
    $combined_order->shipping_address = $shippingAddress;
    $combined_order->billing_address = $combined_order->shipping_address;

    $grand_total = 0; // Initialize grand_total
    $package_number = 1;
    $shop_total = 0;

    $modified_line_items = [];

    foreach ($request->line_items as $line_item) {
      $product_id = $line_item['product_id'];
      $quantity = $line_item['quantity'];

      // Retrieve product details using $product_id
      $product = Product::find($product_id);

      // Calculate the price for this product
      $product_price = $product->lowest_price;

      // Calculate the total price for this line item
      $line_item_total = $product_price * $quantity;

      // Create a new modified line item
      $modified_line_item = $line_item;
      $modified_line_item['price'] = $product_price; // Update the price field in the line item
      $modified_line_item['total'] = $line_item_total; // Update the total field in the line item

      // Add the modified line item to the array
      $modified_line_items[] = $modified_line_item;

      // Add the line item's total to the grand total
      $grand_total += $line_item_total;
    }

    // Replace the original line_items property with the modified line items
    $request->merge(['line_items' => $modified_line_items]);

    // if ($request->delivery_type == 'standard') {
    //     $shipping_cost = $shippingCity->zone->standard_delivery_cost;
    // } elseif ($request->delivery_type == 'express') {
    //     $shipping_cost = $shippingCity->zone->express_delivery_cost;
    // }

    $delivery_type = $request->shipping_lines[0]['method_title']; //$delivery_type = 'express';
    $shipping_cost = floatval($request->shipping_total); //$shipping_cost = 20.00; //$shippingCity->zone->express_delivery_cost;

    $grand_total += $shipping_cost;

    $combined_order->grand_total = $grand_total;
    $combined_order->save();

    // Find the meta_data entry with key "_billing_seller_id"
    $billing_seller_meta = collect($request->meta_data)->firstWhere('key', '_billing_seller_id');

    // Extract the value from the meta_data entry
    $billing_seller_id = $billing_seller_meta['value'];

    // shop order place
    $order = Order::create([
      'shop_id' => $billing_seller_id,
      'combined_order_id' => $combined_order->id,
      'name' => $request->billing['first_name'] . ' ' . $request->billing['last_name'],
      'email' => $request->billing['email'],
      'phone' => $request->billing['phone'],
      'code' => $package_number,
      'shipping_address' => $combined_order->shipping_address,
      'billing_address' => $combined_order->shipping_address,
      'customer_city' => $request->billing['city'],
      'customer_state' => $state->name,
      'shipping_cost' => $shipping_cost,
      'grand_total' => $grand_total,
      'delivery_type' => $delivery_type,
      'payment_type' => $request->payment_method_title,
    ]);

    $package_number++;

    $order_price = $order->grand_total - $order->shipping_cost;

    \Log::info($billing_seller_id);

    $shop_commission = Shop::find($billing_seller_id)->commission;
    \Log::info($shop_commission);
    $seller_earning = $grand_total; // Use grand_total here instead of shop_total
    if ($shop_commission > 0) {
      $seller_earning = ($shop_commission * $order_price) / 100;
    }


    $order->seller_earning = $seller_earning;
    $order->commission_percentage = $shop_commission;
    $order->save();
    OrderUpdate::create([
      'order_id' => $order->id,
      'user_id' => '1',
      'note' => 'Order has been placed.',
    ]);

    // Create OrderDetail records for each product within the order
    foreach ($request->line_items as $line_item) {
      $product_id = $line_item['product_id'];
      $quantity = $line_item['quantity'];

      // Retrieve product details using $product_id
      $product = Product::find($product_id);

      // Calculate the price for this product and add it to the total
      $product_price = $product->lowest_price;
      $shop_total += $product_price * $quantity;

      // Create an OrderDetail record for this product
      $orderDetail = OrderDetail::create([
        'order_id' => $order->id,
        'product_id' => $product_id,
        'price' => $product_price,
        'quantity' => $quantity,
        'total' => $product_price * $quantity,
      ]);
    }

    try {
      //Notification::send($user, new OrderPlacedNotification($combined_order));
      $userEmail = $request->email;
      Notification::route('mail', $userEmail)->notify(new OrderPlacedNotification($combined_order));
      foreach ($combined_order->orders as $order) {
        $seller = $order->orderDetails->first()->product->shop;
        Notification::send($seller, new SellerInvoiceNotification($order));
        Notification::send($seller, new OrderDbNotification($order));
      }
    } catch (\Exception $e) {
    }

    return response()->json([
      'success' => true,
    ]);
  }

  public function paymentDone($combined_order, $payment_method, $payment_info = null)
  {
    foreach ($combined_order->orders as $order) {

      // commission calculation
      calculate_seller_commision($order);

      $order->payment_status = 'paid';
      $order->payment_type = $payment_method;
      $order->payment_details = $payment_info;
      $order->save();
    }
  }

  public function productDownloads()
  {
    $products = DB::table('orders')
      ->orderBy('id', 'desc')
      ->join('order_details', 'orders.id', '=', 'order_details.order_id')
      ->join('products', 'order_details.product_id', '=', 'products.id')
      ->where('orders.user_id', auth()->id())
      ->where('products.digital', '1')
      ->where('orders.payment_status', 'paid')
      ->select('products.*')
      ->paginate(15);

    return DownloadedProductResource::collection($products)->additional([
      'success' => true,
      'status' => 200
    ]);
  }
  public function download($product)
  {
    $product = Product::findOrFail(($product));

    $upload = Upload::findOrFail($product->file_name);

    if (env('FILESYSTEM_DRIVER') == "s3") {
      return \Storage::disk('s3')->download($upload->file_name, $upload->file_original_name . "." . $upload->extension);
    } else {

      if (file_exists(base_path('public/' . $upload->file_name))) {
        return response()->download($_SERVER['HTTP_HOST'] . 'public/' . $upload->file_name);
      }
    }
  }
}
