<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

use App\Models\User;
use App\Models\City;
use App\Models\State;
use App\Models\Region;
use App\Models\Barangay;
use App\Models\SellerPackage;
use App\Models\SellerPackagePayment;

use App\Notifications\SellerRegistrationNotification;
use App\Notifications\SellerReferralRegistrationNotification;

class HomeController extends Controller
{
  public function index(Request $request)
  {
    return view('frontend.home');
  }

  public function signup(Request $request)
  {
    // 1) Validate referrer (sellerid) against users.id
    $selleridValidator = Validator::make(
      $request->only('sellerid'),
      [
        'sellerid' => 'required|exists:users,id',
      ],
      [
        'sellerid.required' => 'This link is invalid. Please check your link and try again.',
        'sellerid.exists'   => 'This link is invalid. Please check your link and try again.',
      ]
    );

    if ($selleridValidator->fails()) {
      $errorMessage = $selleridValidator->errors()->first('sellerid');
      flash(translate($errorMessage))->error();
      return back()->withErrors($selleridValidator)->withInput();
    }

    // 2) Base validation
    $baseValidator = Validator::make(
      $request->all(),
      [
        'first_name' => 'required|string|max:255',
        'last_name'  => 'required|string|max:255',
        'email'      => [
          'required',
          'string',
          'email',
          'max:255',
          'unique:users',
          function ($attribute, $value, $fail) {
            // Gmail / Yahoo / Microsoft only
            if (!preg_match('/^[a-zA-Z0-9._%+-]+@(gmail|yahoo|microsoft)\.(com|net|org)$/i', $value)) {
              $fail('Only Gmail, Yahoo, or Microsoft emails are allowed.');
            }
          },
        ],
        'phone'      => 'required|string|regex:/^[0-9]{11}$/',
        // You changed UI copy earlier (6–8) but validator here is 6–20; align UI text accordingly.
        'password'   => 'required|string|min:6|max:20|confirmed',
        'membership' => 'required|exists:seller_packages,id',
      ],
      [
        'email.unique'       => 'The email address has already been registered. Please use a different email.',
        'email.regex'        => 'Only Gmail, Yahoo, or Microsoft emails are allowed.',
        'password.min'       => 'The password must be at least 6 characters.',
        'password.max'       => 'The password may not be greater than 20 characters.',
        'password.confirmed' => 'The password confirmation does not match.',
      ]
    );

    if ($baseValidator->fails()) {
      return back()->withErrors($baseValidator)->withInput();
    }

    // 3) Load package & determine required locations by geographic_level
    $package = SellerPackage::findOrFail($request->membership);
    $level   = (int) ($package->geographic_level ?? 0);

    // Build conditional rules
    $locationRules = [];
    switch ($level) {
      case 1: // barangay + city
        $locationRules['selected_city']     = 'required|exists:cities,id';
        $locationRules['selected_barangay'] = 'required|exists:barangays,id';
        break;
      case 2: // city + state
        $locationRules['selected_state'] = 'required|exists:states,id';
        $locationRules['selected_city']  = 'required|exists:cities,id';
        break;
      case 3: // state only
        $locationRules['selected_state'] = 'required|exists:states,id';
        break;
      case 4: // region only
        $locationRules['selected_region'] = 'required|exists:regions,id';
        break;
      default: // 0 → no locations required
        break;
    }

    if (!empty($locationRules)) {
      $locValidator = Validator::make($request->all(), $locationRules);
      if ($locValidator->fails()) {
        return back()->withErrors($locValidator)->withInput();
      }
    }

    // 4) Create everything atomically
    DB::beginTransaction();
    try {
      // Create user
      $user = new User();
      $user->name         = trim($request->first_name . ' ' . $request->last_name);
      $user->email        = $request->email;
      $user->phone        = $request->phone;
      $user->user_type    = 'seller';
      $user->referred_by  = $request->sellerid;
      $user->password     = Hash::make($request->password);
      $user->email_verified_at = now()->format('Y-m-d H:i:s'); // use H:i:s (minutes = i)
      $user->save();

      // Package approval & (optional) payment record
      $set_package_payment = null;
      $needsPayment = false;

      if ($level === 0) {
        // No payment upload; just wait for manual approval
        $user->approval = 0;   // pending (make sure the column exists)
        $user->save();
        $needsPayment = false;
      } else {
        // Your existing behavior for paid packages (levels 1–4)
        $user->approval = 0;   // pending until payment approved
        $user->save();

        $payment = new SellerPackagePayment();
        $payment->user_id           = $user->id;
        $payment->seller_package_id = $request->membership;
        $payment->amount            = (float) ($package->amount ?? 0);
        $payment->approval          = 0;
        $payment->save();

        $payment->transaction_id = $payment->id
          . '-' . $user->id
          . '-' . $request->membership
          . '-' . \Illuminate\Support\Str::upper(\Illuminate\Support\Str::random(6));
        $payment->save();

        $set_package_payment = $payment;
        $needsPayment = true;
      }

      // Persist package info on user (if these columns exist)
      $user->verification_status     = 0;              // ⚠️ confirm column exists
      $user->package_id              = $package->id;   // ⚠️ confirm column exists
      $user->seller_type             = strtolower($package->name ?? ''); // ok if column exists
      $user->save();

      // 5) (No direct assignment to locations here; bidding happens later)

      $requestedRegionId   = null;
      $requestedStateId    = null;
      $requestedCityId     = null;
      $requestedBarangayId = null;

      switch ($level) {
        case 1: // barangay + city
          $requestedCityId     = (int) $request->selected_city;
          $requestedBarangayId = (int) $request->selected_barangay;
          break;
        case 2: // city + state
          $requestedStateId = (int) $request->selected_state;
          $requestedCityId  = (int) $request->selected_city;
          break;
        case 3: // state only
          $requestedStateId = (int) $request->selected_state;
          break;
        case 4: // region only
          $requestedRegionId = (int) $request->selected_region;
          break;
        default: // 0 → no location required
          break;
      }

      // If we created a pending payment, attach the requested location IDs
      if ($set_package_payment) {
        $set_package_payment->requested_region_id   = $requestedRegionId;
        $set_package_payment->requested_state_id    = $requestedStateId;
        $set_package_payment->requested_city_id     = $requestedCityId;
        $set_package_payment->requested_barangay_id = $requestedBarangayId;

        // Store the ID (not the name) for the leaf location in seller_assign_location
        $locationId = null;
        switch ($level) {
          case 1: // prefer barangay id, fallback to city id
            $locationId = $requestedBarangayId ?: $requestedCityId;
            break;
          case 2: // city id
            $locationId = $requestedCityId;
            break;
          case 3: // state id
            $locationId = $requestedStateId;
            break;
          case 4: // region id
            $locationId = $requestedRegionId;
            break;
          default:
            $locationId = null;
            break;
        }
        $set_package_payment->seller_assign_location = $locationId; // <-- ID only
        $set_package_payment->save();
      }

      $user->seller_assign_location = null; // keep null until payment claim
      $user->save();

      // 6) Notifications
      try {
        $admin = User::where('user_type', 'admin')->first();
        if ($admin) {
          // Ensure your notification classes accept a User (not Shop)
          $admin->notify(new SellerRegistrationNotification($user));
        }
      } catch (\Throwable $e) {
        // swallow admin notify errors
      }

      try {
        $user->notify(new SellerReferralRegistrationNotification($user));
      } catch (\Throwable $e) {
        // swallow user notify errors
      }

      DB::commit();

      return redirect()
        ->route('thank-you-for-registration')
        ->with('pending_approval_only', $level === 0) // show “wait for approval” screen
        ->with('needs_payment', $needsPayment);       // used by the thank-you blade
    } catch (\Throwable $e) {
      DB::rollBack();
      flash(translate('Something went wrong while creating your account. Please try again.'))->error();
      return back()->withInput();
    }
  }

  public function getStates(Request $request)
  {
    $mempckge = $request->query('mempckge');
    $package = SellerPackage::find($mempckge);
    $geographicLevel = $package?->geographic_level ?? 0;

    // If geographic level does not require states, return empty
    if ($geographicLevel < 2) {
      return response()->json(['states' => []]);
    }

    $states = State::query()
      ->where('status', 1)
      ->when($geographicLevel >= 2, function ($q) {
        // Only allow unassigned if needed
        $q->whereNull('seller_assign');
      })
      ->get();

    return response()->json(['states' => $states]);
  }

  public function getCities(Request $request, $stateId = null)
  {
    $mempckge = $request->query('mempckge');
    $package = SellerPackage::find($mempckge);
    $geographicLevel = $package?->geographic_level ?? 0;

    // If geographic level does not require cities, return empty
    if ($geographicLevel < 1) {
      return response()->json(['cities' => []]);
    }

    $query = City::query()->where('status', 1);

    if ($stateId) {
      // When a state is provided (levels 1 or 2 with Province shown)
      $query->where('state_id', $stateId);
    } else {
      // Level 1 has no Province UI → return all eligible cities
      if ($geographicLevel != 1) {
        // safety guard: only allow no-state fetch at level 1
        return response()->json(['cities' => []]);
      }
      // no extra where here; global list (filtered by assignment below)
    }

    // For level 1, only show unassigned cities
    if ($geographicLevel == 1) {
      $query->whereNull('seller_assign');
    }

    $cities = $query->orderBy('name')->get();

    return response()->json(['cities' => $cities]);
  }

  public function getBarangays(Request $request, $cityId)
  {
    $mempckge = $request->query('mempckge');
    $package = SellerPackage::find($mempckge);
    $geographicLevel = $package?->geographic_level ?? 0;

    // If geographic level does not require barangays, return empty
    if ($geographicLevel != 1) {
      return response()->json(['barangays' => []]);
    }

    $barangays = Barangay::query()
      ->where('city_id', $cityId)
      ->where('status', 1)
      ->whereNull('seller_assign')
      ->get();

    return response()->json(['barangays' => $barangays]);
  }

  public function getRegions(Request $request)
  {
    $mempckge = $request->query('mempckge');
    $package = SellerPackage::find($mempckge);
    $geographicLevel = $package?->geographic_level ?? 0;

    if ($geographicLevel != 4) {
      return response()->json(['regions' => []]);
    }

    $regions = Region::where('status', 1)
      ->whereNull('seller_assign')
      ->get();

    return response()->json(['regions' => $regions]);
  }

  public function membershipregistration(Request $request)
  {
    $mempckge = $request->query('mempckge');

    $packages = SellerPackage::all();

    $selectedPackage = null;
    $geographicLevel = 0;

    if ($mempckge) {
      $selectedPackage = SellerPackage::find($mempckge);
      $geographicLevel = $selectedPackage?->geographic_level ?? 0;
    }

    // load states/cities/barangays only if needed
    $states = $cities = $barangays = $regions = collect();

    if ($geographicLevel > 0) {
      $states = State::where('status', 1)->get();
    }

    if ($request->has('selected_state') && $geographicLevel >= 1) {
      $cities = City::where('status', 1)
        ->where('state_id', $request->input('selected_state'))
        ->get();
    }

    if ($request->has('selected_city') && $geographicLevel >= 1) {
      $barangays = Barangay::where('city_id', $request->input('selected_city'))->get();
    }

    if ($geographicLevel == 4) {
      $regions = Region::where('status', 1)->get();
    }

    return view('frontend.membership', compact('packages', 'states', 'cities', 'barangays', 'regions', 'geographicLevel'));
  }

  public function thankyouforregistration(Request $request)
  {
    return view('frontend.thankyouforregistration');
  }
}
