<?php

namespace App\Http\Controllers\Merchant;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\Product;
use App\Models\CommissionHistory;

class DashboardController extends Controller
{
  public function index()
  {
    $uid = auth()->id();

    // Base stats
    $totalProducts = Product::where('user_id', $uid)->count();

    $grossSales = OrderDetail::whereHas('product', fn($q) => $q->where('user_id', $uid))
      ->whereHas('order', fn($q) => $q->where('delivery_status', '!=', 'cancelled'))
      ->sum('total');

    // Orders that contain at least one of this merchant's products
    $orderIdsForMe = OrderDetail::whereHas('product', fn($q) => $q->where('user_id', $uid))
      ->pluck('order_id')
      ->unique();

    // Total distinct orders touching this merchant
    $totalOrders = Order::whereIn('id', $orderIdsForMe)->count();

    // Earnings (all-time) from your ledger
    $earnings = CommissionHistory::where('user_id', $uid)->sum('seller_earning');

    // Delivery status map (handles your mixed naming)
    $statusMap = [
      'placed'            => ['order_placed', 'order placed', 'placed'],
      'packed'            => ['order_packed', 'order packed', 'packed'],
      'shipped'           => ['shipped_to_courier', 'shipped to courier'],
      'in_transit'        => ['in_transit', 'in transit'],
      'out_for_delivery'  => ['out_for_delivery', 'out for delivery'],
      'delivered'         => ['delivered'],
    ];

    // Counts per delivery status (scoped to this merchant's orders)
    $countPlaced         = Order::whereIn('id', $orderIdsForMe)->whereIn('delivery_status', $statusMap['placed'])->count();
    $countPacked         = Order::whereIn('id', $orderIdsForMe)->whereIn('delivery_status', $statusMap['packed'])->count();
    $countShipped        = Order::whereIn('id', $orderIdsForMe)->whereIn('delivery_status', $statusMap['shipped'])->count();
    $countInTransit      = Order::whereIn('id', $orderIdsForMe)->whereIn('delivery_status', $statusMap['in_transit'])->count();
    $countOutForDelivery = Order::whereIn('id', $orderIdsForMe)->whereIn('delivery_status', $statusMap['out_for_delivery'])->count();
    $countDelivered      = Order::whereIn('id', $orderIdsForMe)->whereIn('delivery_status', $statusMap['delivered'])->count();

    // Top products
    $topProducts = Product::where('user_id', $uid)
      ->where('published', 1)
      ->where('approved', 1)
      ->orderByDesc('num_of_sale')
      ->limit(10)
      ->get();

    return view('backend.merchants.dashboard', compact(
      'totalProducts',
      'grossSales',
      'totalOrders',
      'earnings',
      'topProducts',
      'countPlaced',
      'countPacked',
      'countShipped',
      'countInTransit',
      'countOutForDelivery',
      'countDelivered'
    ));
  }
}
