<?php

namespace App\Http\Controllers\Merchant;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderDetail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class EarningController extends Controller
{
  public function index(Request $request)
  {
    $uid = auth()->id();

    // Orders that contain this merchant's products
    $orderIdsForMe = OrderDetail::whereHas('product', fn($q) => $q->where('user_id', $uid))
      ->pluck('order_id')->unique();

    // --- existing totals you already have (kept as-is) ---
    $ordersCount = Order::whereIn('id', $orderIdsForMe)->count();

    $ordersAmountAll = (float) OrderDetail::whereHas('product', fn($q) => $q->where('user_id', $uid))
      ->sum('total');

    $completedIds = Order::whereIn('id', $orderIdsForMe)
      ->where('payment_status', 'paid')
      ->pluck('id');

    $completedCount = $completedIds->count();

    $grossPaid = (float) OrderDetail::whereHas('product', fn($q) => $q->where('user_id', $uid))
      ->whereIn('order_id', $completedIds)
      ->sum('total');

    $adminEarning = $completedCount > 0 ? round($grossPaid * 0.10, 2) : 0.0;

    // keep your current Net logic (or the simplified one you preferred earlier)
    $net = $completedCount > 0 ? max(0, $grossPaid - $adminEarning) : 0.0;

    // Per-product (PAID only)
    $perProduct = OrderDetail::with('product:id,name,user_id')
      ->whereHas('product', fn($q) => $q->where('user_id', $uid))
      ->whereIn('order_id', $completedIds)
      ->selectRaw('product_id, SUM(total) as gross, SUM(quantity) as qty')
      ->groupBy('product_id')
      ->orderByDesc(DB::raw('SUM(total)'))
      ->paginate(15);

    // ---------- NEW: Orders table (with date range) ----------
    $date_range = $request->get('date_range');
    $ordersQuery = Order::with(['combined_order', 'user', 'orderDetails'])
      ->whereIn('id', $orderIdsForMe);

    if (!empty($date_range) && str_contains($date_range, ' to ')) {
      [$from, $to] = array_map('trim', explode(' to ', $date_range, 2));
      // inclusive range
      $ordersQuery->whereDate('created_at', '>=', $from)
        ->whereDate('created_at', '<=', $to);
    }

    // Paginated orders list
    $orders = $ordersQuery->latest('id')->paginate(15)->appends($request->query());

    // Per-order stats specific to THIS merchant (count & amount)
    $stats = OrderDetail::whereIn('order_id', $orders->pluck('id'))
      ->whereHas('product', fn($q) => $q->where('user_id', $uid))
      ->selectRaw('order_id, COUNT(*) as items_count, SUM(total) as amount')
      ->groupBy('order_id')
      ->get()
      ->keyBy('order_id');

    return view('backend.merchants.earnings.index', [
      'ordersCount'     => $ordersCount,
      'ordersAmountAll' => $ordersAmountAll,
      'completedCount'  => $completedCount,
      'gross'           => $grossPaid,
      'adminEarning'    => $adminEarning,
      'net'             => $net,
      'perProduct'      => $perProduct,

      // new
      'orders'          => $orders,
      'orderStats'      => $stats,
      'date_range'      => $date_range,
    ]);
  }
}
