<?php

namespace App\Http\Controllers\Merchant;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Services\WooCommerceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ProductController extends Controller
{
  public function __construct(private WooCommerceService $wc) {}

  public function index()
  {
    $products = Product::where('user_id', auth()->id())->latest()->paginate(15);
    return view('backend.merchants.products.index', compact('products'));
  }

  public function create()
  {
    return view('backend.merchants.products.create');
  }

  public function store(Request $request)
  {
    $validated = $request->validate([
      'name'               => 'required|string|max:200',
      'unit'               => 'nullable|string|max:20',
      'thumbnail_img'      => 'nullable|string|max:255',
      'price'              => 'required|numeric|min:0',
      'sku'                => 'nullable|string|max:100',
      'stock'              => 'required|in:0,1',         // 1=in stock, 0=out of stock
      'description'        => 'nullable|string',
      'short_description'  => 'nullable|string',
      'status'             => 'required|in:0,1',         // 1=publish, 0=draft
      'tags'               => 'nullable|string|max:2000' // comma-separated
    ]);

    // Save locally first (your current behavior)
    $p = new Product();
    $p->user_id           = auth()->id();
    $p->name              = $validated['name'];
    $p->unit              = $validated['unit'] ?? null;
    $p->min_qty           = 1;
    $p->thumbnail_img     = $validated['thumbnail_img'] ?? null;
    $p->lowest_price      = (float) $validated['price'];  // single price path
    $p->highest_price     = (float) $validated['price'];
    $p->sku               = $validated['sku'] ?? null;
    $p->stock             = (int) $validated['stock'];
    $p->description       = $validated['description'] ?? null;
    $p->short_description = $validated['short_description'] ?? null;
    $p->published         = (int) $validated['status'];
    $p->approved          = 0;
    $p->tags              = $validated['tags'] ?? null;
    $p->save();

    // Sync to WooCommerce (CREATE)
    try {
      $payload = $this->toWooPayload($p);
      $res = $this->wc->createProduct($payload);

      if (is_array($res) && isset($res['id'])) {
        $p->woocommerce_id = $res['id'];
        $p->save();
      } else {
        Log::warning('WooCommerce create returned unexpected response', ['res' => $res, 'local_id' => $p->id]);
      }
    } catch (\Throwable $e) {
      Log::error('WooCommerce create failed', ['local_id' => $p->id, 'error' => $e->getMessage()]);
      // keep local product; just log the failure
    }

    flash(translate('Product created successfully'))->success();
    return redirect()->route('merchant.products.index');
  }

  public function edit(Product $product)
  {
    $this->authorize('update', $product);
    return view('backend.merchants.products.edit', compact('product'));
  }

  public function update(Request $request, Product $product)
  {
    $this->authorize('update', $product);

    $validated = $request->validate([
      'name'               => 'required|string|max:200',
      'unit'               => 'nullable|string|max:20',
      'thumbnail_img'      => 'nullable|string|max:255',
      'price'              => 'required|numeric|min:0',
      'sku'                => 'nullable|string|max:100',
      'stock'              => 'required|in:0,1',
      'description'        => 'nullable|string',
      'short_description'  => 'nullable|string',
      'status'             => 'required|in:0,1',
      'tags'               => 'nullable|string|max:2000',
    ]);

    // Update locally first (your current behavior)
    $product->fill([
      'name'              => $validated['name'],
      'unit'              => $validated['unit'] ?? null,
      'min_qty'           => 1,
      'thumbnail_img'     => $validated['thumbnail_img'] ?? $product->thumbnail_img,
      'lowest_price'      => (float) $validated['price'],
      'highest_price'     => (float) $validated['price'],
      'sku'               => $validated['sku'] ?? null,
      'stock'             => (int) $validated['stock'],
      'description'       => $validated['description'] ?? null,
      'short_description' => $validated['short_description'] ?? null,
      'published'         => (int) $validated['status'],
      'tags'              => $validated['tags'] ?? null,
    ])->save();

    // Sync to WooCommerce (UPDATE; or CREATE if missing WC id)
    try {
      $payload = $this->toWooPayload($product);

      if ($product->woocommerce_id) {
        $this->wc->updateProduct($product->woocommerce_id, $payload);
      } else {
        $res = $this->wc->createProduct($payload);
        if (is_array($res) && isset($res['id'])) {
          $product->woocommerce_id = $res['id'];
          $product->save();
        }
      }
    } catch (\Throwable $e) {
      Log::error('WooCommerce update/create failed', [
        'local_id' => $product->id,
        'wc_id'    => $product->woocommerce_id,
        'error'    => $e->getMessage()
      ]);
    }

    flash(translate('Product updated successfully'))->success();
    return redirect()->route('merchant.products.index');
  }

  public function destroy(Product $product)
  {
    $this->authorize('delete', $product);

    // Sync to WooCommerce (DELETE) – ignore if it fails, still remove locally
    try {
      if ($product->woocommerce_id) {
        $this->wc->deleteProduct($product->woocommerce_id);
      }
    } catch (\Throwable $e) {
      Log::error('WooCommerce delete failed', [
        'local_id' => $product->id,
        'wc_id'    => $product->woocommerce_id,
        'error'    => $e->getMessage()
      ]);
    }

    $product->delete();
    flash(translate('Product deleted successfully'))->success();
    return back();
  }

  /**
   * Map local Product -> WooCommerce payload
   */
  private function toWooPayload(Product $p): array
  {
    // Build a public URL for the thumbnail (Woo needs a reachable URL)
    $thumbUrl = null;
    try {
      if (!empty($p->thumbnail_img)) {
        $thumbUrl = function_exists('uploaded_asset') ? uploaded_asset($p->thumbnail_img) : null;
      }
    } catch (\Throwable $e) {
      $thumbUrl = null;
    }

    // Convert comma-separated tags to WC tag objects
    $tags = [];
    if (!empty($p->tags)) {
      foreach (array_filter(array_map('trim', explode(',', $p->tags))) as $tag) {
        $tags[] = ['name' => $tag];
      }
    }

    return [
      'name'              => (string) $p->name,
      'type'              => 'simple',
      'status'            => $p->published ? 'publish' : 'draft',
      'regular_price'     => (string) number_format((float) ($p->lowest_price ?? 0), 2, '.', ''),
      'sku'               => $p->sku ?: null,
      // If you don't manage exact quantities yet, leave manage_stock false and set stock_status only
      'manage_stock'      => false,
      'stock_status'      => $p->stock ? 'instock' : 'outofstock',
      'short_description' => $p->short_description ?? '',
      'description'       => $p->description ?? '',
      'tags'              => $tags,
      'images'            => $thumbUrl ? [['src' => $thumbUrl]] : [],
    ];
  }
}
