<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Services\WooCommerceService;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductTranslation;
use App\Models\ProductVariation;
use App\Models\ProductVariationCombination;
use App\Models\ProductAttribute;
use App\Models\ProductAttributeValue;
use App\Models\Attribute;
use App\Models\AttributeValue;
use App\Models\Category;
use App\Models\ProductTax;
use App\Models\ShopBrand;
use App\Models\User;
use App\Utility\CategoryUtility;
use CoreComponentRepository;
use Artisan;
use Carbon\Carbon;

class ProductController extends Controller
{
  public function __construct()
  {
    $this->middleware(['permission:show_products'])->only('index');
    $this->middleware(['permission:add_products'])->only('create');
    $this->middleware(['permission:view_products'])->only('show');
    $this->middleware(['permission:edit_products'])->only('edit');
    $this->middleware(['permission:duplicate_products'])->only('duplicate');
    $this->middleware(['permission:delete_products'])->only('destroy');
  }

  /**
   * Display a listing of the resource.
   *
   * @return \Illuminate\Http\Response
   */

  public function index(Request $request)
  {
    $col_name = null;
    $query = null;
    $sort_search = null;
    // $products = Product::orderBy('created_at', 'desc')->where('shop_id', auth()->user()->shop_id);
    $products = Product::orderBy('created_at', 'desc');

    if ($request->search != null) {
      $products = $products->where('name', 'like', '%' . $request->search . '%');
      $sort_search = $request->search;
    }
    if ($request->type != null) {
      $var = explode(",", $request->type);
      $col_name = $var[0];
      $query = $var[1];
      $products = $products->orderBy($col_name, $query);
      $sort_type = $request->type;
    }

    $products = $products->paginate(15);
    $type = 'All';

    return view('backend.product.products.index', compact('products', 'type', 'col_name', 'query', 'sort_search'));
  }


  /**
   * Show the form for creating a new resource.
   *
   * @return \Illuminate\Http\Response
   */
  public function create()
  {
    $categories = Category::where('level', 0)->where('digital', 0)->get();
    // $attributes = Attribute::get();
    return view('backend.product.products.create', compact('categories'));
  }

  public function store(Request $request, WooCommerceService $wc)
  {
    // CoreComponentRepository::instantiateShopRepository();

    // 1. Save product locally
    $product = new Product;
    $product->name = $request->name;
    $product->unit = $request->unit;
    $product->min_qty = 1;
    $product->max_qty = 0;
    $product->thumbnail_img = $request->thumbnail_img;
    $product->description = $request->description;
    $product->short_description = $request->short_description;
    $product->published = $request->status;
    $product->stock = $request->stock;
    $product->slug = Str::slug($request->name, '-') . '-' . strtolower(Str::random(5));
    $product->lowest_price = $request->price;
    $product->highest_price = $request->price;
    $product->sku = $request->sku;
    $product->save();

    try {
      // 2. Push product to WooCommerce
      $wcResponse = $wc->createProduct([
        'name' => $product->name,
        'type' => 'simple',
        'regular_price' => (string) $product->lowest_price,
        'description' => $product->description,
        'short_description' => $request->short_description ?? '',
        'sku' => $product->sku,
        'manage_stock' => false,
        'stock_status' => $request->stock == '1' ? 'instock' : 'outofstock',
        // Uncomment this when running on production server
        // 'images' => [
        //     ['src' => $product->thumbnail_img],
        // ],
      ]);

      // 3. Save WooCommerce product ID
      if (isset($wcResponse['id'])) {
        $product->woocommerce_id = $wcResponse['id'];
        $product->save();
      } else {
        Log::error('WooCommerce Create Product Failed', ['response' => $wcResponse]);
      }
    } catch (\Exception $e) {
      Log::error('WooCommerce Create Product Exception', ['error' => $e->getMessage()]);
    }

    flash(translate('Product created and synced with WooCommerce!'))->success();
    return redirect()->route('product.index');
  }

  /**
   * Display the specified resource.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function show($id)
  {
    return view('backend.product.products.show', [
      'product' => Product::with('variations.combinations')->findOrFail($id)
    ]);
  }

  /**
   * Show the form for editing the specified resource.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function edit(Request $request, $id)
  {
    $product = Product::findOrFail($id);
    if ($product->user_id != auth()->user()->user_id) {
      abort(403);
    }

    $lang = $request->lang;
    //$categories = Category::where('level', 0)->where('digital', 0)->get();
    //$all_attributes = Attribute::get();
    return view('backend.product.products.edit', compact('product', 'lang'));
  }

  public function update(Request $request, $id, WooCommerceService $wc)
  {
    $product = Product::findOrFail($id);

    // 1. Update product locally
    $product->name = $request->name;
    $product->unit = $request->unit;
    $product->description = $request->description;
    $product->short_description = $request->short_description;
    $product->min_qty = 1;
    $product->max_qty = 0;
    $product->thumbnail_img = $request->thumbnail_img;
    $product->published = $request->status;
    $product->lowest_price = $request->price;
    $product->highest_price = $request->price;
    $product->sku = $request->sku;
    $product->stock = $request->stock; // ✅ Save stock locally
    $product->slug = (!is_null($request->slug))
      ? Str::slug($request->slug, '-')
      : Str::slug($request->name, '-') . '-' . strtolower(Str::random(5));
    $product->save();

    try {
      if ($product->woocommerce_id) {
        // 2. Update existing product in WooCommerce
        $wc->updateProduct($product->woocommerce_id, [
          'name' => $product->name,
          'regular_price' => (string) $product->lowest_price,
          'description' => $product->description,
          'short_description' => $request->short_description ?? '',
          'sku' => $product->sku,

          // ✅ Sync stock with WooCommerce
          'manage_stock' => false, // we're using dropdown status only
          'stock_status' => $request->stock == '1' ? 'instock' : 'outofstock',

          // Uncomment when on production with public images
          // 'images' => [
          //     ['src' => $product->thumbnail_img],
          // ],
        ]);
      } else {
        // 3. If not yet synced, create it on WooCommerce
        $wcResponse = $wc->createProduct([
          'name' => $product->name,
          'type' => 'simple',
          'regular_price' => (string) $product->lowest_price,
          'description' => $product->description,
          'short_description' => $request->short_description ?? '',
          'sku' => $product->sku,

          // ✅ Sync stock for new products
          'manage_stock' => false,
          'stock_status' => $request->stock == '1' ? 'instock' : 'outofstock',

          // Uncomment when on production with public images
          // 'images' => [
          //     ['src' => $product->thumbnail_img],
          // ],
        ]);

        if (isset($wcResponse['id'])) {
          $product->woocommerce_id = $wcResponse['id'];
          $product->save();
        } else {
          Log::error('WooCommerce Create Product Failed', ['response' => $wcResponse]);
        }
      }
    } catch (\Exception $e) {
      Log::error('WooCommerce Update Product Exception', ['error' => $e->getMessage()]);
    }

    flash(translate('Product updated and synced with WooCommerce!'))->success();
    return redirect()->route('product.index');
  }

  public function destroy($id, WooCommerceService $wc)
  {
    $product = Product::findOrFail($id);

    try {
      // 1. Delete product from WooCommerce if synced
      if ($product->woocommerce_id) {
        $wc->deleteProduct($product->woocommerce_id);
      }
    } catch (\Exception $e) {
      Log::error('WooCommerce Delete Product Exception', ['error' => $e->getMessage()]);
    }

    // 2. Delete locally
    $product->product_translations()->delete();
    $product->variations()->delete();
    $product->variation_combinations()->delete();
    $product->reviews()->delete();
    $product->product_categories()->delete();
    $product->offers()->delete();
    $product->wishlists()->delete();
    $product->attributes()->delete();
    $product->attribute_values()->delete();
    $product->taxes()->delete();
    $product->delete();

    flash(translate('Product deleted successfully!'))->success();
    return redirect()->route('product.index');
  }


  // public function destroy($id)
  // {
  //   $product = Product::findOrFail($id);
  //   $product->product_translations()->delete();
  //   $product->variations()->delete();
  //   $product->variation_combinations()->delete();
  //   $product->reviews()->delete();
  //   $product->product_categories()->delete();
  //   //$product->carts()->delete();
  //   $product->offers()->delete();
  //   $product->wishlists()->delete();
  //   $product->attributes()->delete();
  //   $product->attribute_values()->delete();
  //   $product->taxes()->delete();

  //   if (Product::destroy($id)) {
  //     flash(translate('Product has been deleted successfully'))->success();
  //     return redirect()->route('product.index');
  //   } else {
  //     flash(translate('Something went wrong'))->error();
  //     return back();
  //   }
  // }

  /**
   * Duplicates the specified resource from storage.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function duplicate(Request $request, $id)
  {
    $product = Product::find($id);
    $product_new = $product->replicate();
    $product_new->slug = Str::slug($product_new->name, '-') . '-' . strtolower(Str::random(5));

    if ($product_new->save()) {

      // variation duplicate
      foreach ($product->variations as $key => $variation) {
        $p_variation = new ProductVariation;
        $p_variation->product_id = $product_new->id;
        $p_variation->code = $variation->code;
        $p_variation->price = $variation->price;
        $p_variation->stock = $variation->stock;
        $p_variation->sku = $variation->sku;
        $p_variation->img = $variation->img;
        $p_variation->save();

        // variation combination duplicate
        foreach ($variation->combinations as $key => $combination) {
          $p_variation_comb = new ProductVariationCombination;
          $p_variation_comb->product_id = $product_new->id;
          $p_variation_comb->product_variation_id = $p_variation->id;
          $p_variation_comb->attribute_id = $combination->attribute_id;
          $p_variation_comb->attribute_value_id = $combination->attribute_value_id;
          $p_variation_comb->save();
        }
      }

      // attribute duplicate
      foreach ($product->attributes as $key => $attribute) {
        $p_attribute = new ProductAttribute;
        $p_attribute->product_id = $product_new->id;
        $p_attribute->attribute_id = $attribute->attribute_id;
        $p_attribute->save();
      }

      // attribute value duplicate
      foreach ($product->attribute_values as $key => $attribute_value) {
        $p_attr_value = new ProductAttributeValue;
        $p_attr_value->product_id = $product_new->id;
        $p_attr_value->attribute_id = $attribute_value->attribute_id;
        $p_attr_value->attribute_value_id = $attribute_value->attribute_value_id;
        $p_attr_value->save();
      }

      // translation duplicate
      foreach ($product->product_translations as $key => $translation) {
        $product_translation = new ProductTranslation;
        $product_translation->product_id = $product_new->id;
        $product_translation->name = $translation->name;
        $product_translation->unit = $translation->unit;
        $product_translation->description = $translation->description;
        $product_translation->lang = $translation->lang;
        $product_translation->save();
      }

      //categories duplicate
      foreach ($product->product_categories as $key => $category) {
        $p_category = new ProductCategory;
        $p_category->product_id = $product_new->id;
        $p_category->category_id = $category->category_id;
        $p_category->save();
      }

      // taxes duplicate
      foreach ($product->taxes as $key => $tax) {
        $p_tax = new ProductTax;
        $p_tax->product_id = $product_new->id;
        $p_tax->tax_id = $tax->tax_id;
        $p_tax->tax = $tax->tax;
        $p_tax->tax_type = $tax->tax_type;
        $p_tax->save();
      }

      flash(translate('Product has been duplicated successfully'))->success();
      return redirect()->route('product.index');
    } else {
      flash(translate('Something went wrong'))->error();
      return back();
    }
  }

  public function get_products_by_subcategory(Request $request)
  {
    $products = Product::where('subcategory_id', $request->subcategory_id)->get();
    return $products;
  }

  public function get_products_by_brand(Request $request)
  {
    $products = Product::where('brand_id', $request->brand_id)->get();
    return view('partials.product_select', compact('products'));
  }

  public function updatePublished(Request $request)
  {
    $product = Product::findOrFail($request->id);
    $product->published = $request->status;
    $product->save();

    cache_clear();

    return 1;
  }

  public function sku_combination(Request $request)
  {
    // dd($request->all());

    $option_choices = array();

    if ($request->has('product_options')) {
      $product_options = $request->product_options;
      sort($product_options, SORT_NUMERIC);

      foreach ($product_options as $key => $option) {

        $option_name = 'option_' . $option . '_choices';
        $choices = array();

        if ($request->has($option_name)) {

          $product_option_values = $request[$option_name];
          sort($product_option_values, SORT_NUMERIC);

          foreach ($product_option_values as $key => $item) {
            array_push($choices, $item);
          }
          $option_choices[$option] = $choices;
        }
      }
    }

    $combinations = array(array());
    foreach ($option_choices as $property => $property_values) {
      $tmp = array();
      foreach ($combinations as $combination_item) {
        foreach ($property_values as $property_value) {
          $tmp[] = $combination_item + array($property => $property_value);
        }
      }
      $combinations = $tmp;
    }

    // dd($option_choices,$combinations);

    return view('backend.product.products.sku_combinations', compact('combinations'))->render();
  }

  public function new_attribute(Request $request)
  {
    $attributes = Attribute::query();
    if ($request->has('product_attributes')) {
      foreach ($request->product_attributes as $key => $value) {
        if ($value == NULL) {
          return array(
            'count' => -1,
            'view' => view('backend.product.products.new_attribute', compact('attributes'))->render(),
          );
        }
      }
      $attributes->whereNotIn('id', array_diff($request->product_attributes, [null]));
    }

    $attributes = $attributes->get();

    return array(
      'count' => count($attributes),
      'view' => view('backend.product.products.new_attribute', compact('attributes'))->render(),
    );
  }

  public function get_attribute_values(Request $request)
  {

    $attribute_id = $request->attribute_id;
    $attribute_values = AttributeValue::where('attribute_id', $attribute_id)->get();

    return view('backend.product.products.new_attribute_values', compact('attribute_values', 'attribute_id'));
  }

  public function new_option(Request $request)
  {

    $attributes = Attribute::query();
    if ($request->has('product_options')) {
      foreach ($request->product_options as $key => $value) {
        if ($value == NULL) {
          return array(
            'count' => -1,
            'view' => view('backend.product.products.new_option', compact('attributes'))->render(),
          );
        }
      }
      $attributes->whereNotIn('id', array_diff($request->product_options, [null]));
      if (count($request->product_options) === 3) {
        return array(
          'count' => -2,
          'view' => view('backend.product.products.new_option', compact('attributes'))->render(),
        );
      }
    }

    $attributes = $attributes->get();

    return array(
      'count' => count($attributes),
      'view' => view('backend.product.products.new_option', compact('attributes'))->render(),
    );
  }

  public function get_option_choices(Request $request)
  {

    $attribute_id = $request->attribute_id;
    $attribute_values = AttributeValue::where('attribute_id', $attribute_id)->get();

    return view('backend.product.products.new_option_choices', compact('attribute_values', 'attribute_id'));
  }
  public function updateProductApproval(Request $request)
  {
    $product = Product::findOrFail($request->id);
    $product->approved = $request->approved;
    $shop = $product->shop;

    if ($shop->user->user_type == 'seller') {
      if (
        $shop->package_invalid_at == null
        || Carbon::now()->diffInDays(Carbon::parse($shop->package_invalid_at), false) < 0
        || $shop->product_upload_limit <= $shop->products()->where('published', 1)->count()
      ) {
        return 0;
      }
    }
    $product->save();
    cache_clear();
    return 1;
  }
}
