<?php

namespace App\Jobs;

use App\Models\Order;
use App\Models\CommissionHistory;
use App\Models\SellerPackage;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessOrderCommission implements ShouldQueue
{
  use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

  public $order;

  public function __construct(Order $order)
  {
    // preload relationships you’ll need
    $this->order = $order->load(['user', 'orderDetails']);
  }

  public function handle()
  {
    $order = $this->order;

    if ($this->shouldSkipCommission($order)) {
      return;
    }

    // Commission
    $this->reseller_commission($order);

    // Bonuses
    $this->calculateAndCreateAreaSaleBonus($order);
  }

  private function reseller_commission(Order $order)
  {
    $user = $order->user;
    $seller_package = SellerPackage::find($user->package_id);

    $order_price = $order->grand_total - $order->shipping_cost;

    if ($seller_package->commission_type === 'percent') {
      $seller_earning = ($seller_package->commission * $order_price) / 100;
    } else {
      $totalQuantity = $order->orderDetails->sum('quantity');
      $seller_earning = $totalQuantity * $seller_package->commission;
    }

    $user->current_balance += $seller_earning;

    CommissionHistory::create([
      'order_id'       => $order->id,
      'user_id'        => $user->id,
      'seller_earning' => $seller_earning,
      'details'        => 'Sale Commission from Order - ' . $order->name,
      'source_type'    => 'order_sale',
    ]);

    $order->save();
    $user->save();
  }

  private function calculateAndCreateAreaSaleBonus(Order $order)
  {
    if ($order->user && $order->user->referred_by) {
      $this->processReferralSaleBonus($order->user->referred_by, $order);
    }

    // CITY LEVEL BONUS
    if ($order->city && $order->city->seller_assign) {
      $this->processAreaSaleBonus($order->city->seller_assign, $order);
    }

    // STATE LEVEL BONUS
    if ($order->state && $order->state->seller_assign) {
      $this->processAreaSaleBonus($order->state->seller_assign, $order);
    }

    // REGION LEVEL BONUS
    if ($order->region && $order->region->seller_assign) {
      $this->processAreaSaleBonus($order->region->seller_assign, $order);
    }
  }

  private function processReferralSaleBonus($referrerToCheck, Order $order)
  {
    $referrerUser = User::find($referrerToCheck);

    if (!$referrerUser || !$referrerUser->package_id) {
      return;
    }

    $sellerPackage = SellerPackage::find($referrerUser->package_id);

    if ($sellerPackage->ref_sale_commission_type === 'percent') {
      $orderSubtotal = $order->orderDetails->sum('total');
      $sellerEarning = ($orderSubtotal * $sellerPackage->ref_sale_commission) / 100;
    } else {
      $totalQuantity = $order->orderDetails->sum('quantity');
      $sellerEarning = $totalQuantity * $sellerPackage->ref_sale_commission;
      $sourceType = 'referral_sale';
    }

    $this->createCommissionHistory($referrerToCheck, $sellerEarning, $sourceType, 'Referral Sale Commission.');
  }

  private function processAreaSaleBonus($sellerAssign, Order $order)
  {
    $user = User::find($sellerAssign);

    if (!$user || $user->approval != 1) {
      return;
    }

    $sellerPackage = SellerPackage::find($user->package_id);

    if ($sellerPackage->area_sale_bonus_type === 'percent') {
      $orderSubtotal = $order->orderDetails->sum('total');
      $sellerEarning = ($orderSubtotal * $sellerPackage->area_sale_bonus) / 100;
    } else {
      $totalQuantity = $order->orderDetails->sum('quantity');
      $sellerEarning = $totalQuantity * $sellerPackage->area_sale_bonus;
      $sourceType = 'area_bonus';
    }

    $this->createCommissionHistory($sellerAssign, $sellerEarning, $sourceType, 'Area Sale Bonus.');
  }

  private function createCommissionHistory($userId, $sellerEarning, $details, $sourceType)
  {
    CommissionHistory::create([
      'user_id'        => $userId,
      'seller_earning' => $sellerEarning,
      'details'        => $details,
      'source_type'    => $sourceType,
    ]);

    $user = User::findOrFail($userId);
    $user->current_balance += $sellerEarning;
    $user->save();
  }

  private function shouldSkipCommission(Order $order): bool
  {
    return !Order::where('id', $order->id)
      ->whereNull('code')
      ->exists();
  }
}
